/**
 * @license
 * Copyright 2022 Vivid Inc. and/or its affiliates.
 */

"use strict";

if (typeof VT == 'undefined') {
    var VT = {};
}
if (typeof VT.Components == 'undefined') {
    VT.Components = {};
}

// TODO _.debounce( ... ) all user input. 250ms ?
// TODO dampen draw hz when currentIssueKey is the same. work properly with using arrow keys quickly in distance field.

VT.Components.FETCH_IRG_EVENT = 'VividTraceFetchIRG';

VT.Components.REDRAW_IRG_EVENT = 'VividTraceRedrawIRG';

/**
 * @param arg a string that might name one or more issue keys, separated by whitespace and/or commas
 * @returns either an array of issue keys, or undefined
 */
VT.Components.asIssueKeys = function(arg) {
    var candidateIssueKeys = _.filter(
        arg.split(/[,\s]+/),
        function (x) {
            return _.size(x) >= 1;
        }
    );
    var isIssueKeys = _.every(
        candidateIssueKeys,
        function(x) {
            // Note: The regex is defined by the rules documented at these sources:
            // https://confluence.atlassian.com/jira/changing-the-project-key-format-192534.html
            // file://vivid-trace-jira-addon/src/main/antlr4/vivid/trace/jql/grammar/RelationsParameter.g4
            // https://vivid-inc.net/en/vivid-trace/docs/1.4/administration.html#limitations
            return /^([0-9]+)|([a-zA-Z][a-zA-Z0-9_]*\-[0-9]+)$/.test(x);
        }
    );
    return isIssueKeys ? candidateIssueKeys : undefined;
};

VT.Components.auiDisableButton = function(sel) {
    AJS.$(sel)
        .attr({
            'aria-disabled': 'true',
            disabled: 'disabled'
        })
        .addClass('disabled');
};

VT.Components.auiEnableButton = function(sel) {
    AJS.$(sel)
        .removeAttr('aria-disabled disabled')
        .removeClass('disabled');
};

VT.Components.DistanceSettingView = VT.Backbone.View.extend({
    clickDistanceButton: function(event) {
        event.preventDefault();
        var v = event.target.value;
        this.setDistance(v);
        this.updateDistanceField();
        return false;
    },
    changeDistanceKeyDown: function(event) {
        var key = event.keyCode;
        var d = this.getValueAsInt(event);
        if (key === 38 || key === 104) {
            event.preventDefault();
            this.setDistance(isNaN(d) ? '0' : String(d + 1));
            this.updateDistanceField()
                .select();
            // Prevent the cursor keys from changing the menu item selection.
            return false;
        } else if (key === 40 || key === 98) {
            event.preventDefault();
            this.setDistance(isNaN(d) ? '' : (d >= 1 ? String(d - 1) : '0'));
            this.updateDistanceField()
                .select();
            // Prevent the cursor keys from changing the menu item selection.
            return false;
        }
    },
    changeDistanceKeyUp: function(event) {
        var key = event.keyCode;
        var v = event.target.value;
        // Ignore keys specifically handled in changeDistanceKeyDown()
        if (_.indexOf([38, 40, 98, 104], key) < 0) {
            this.setDistance(v);
        }
    },
    getValueAsInt: function(event) {
        // parseInt() accepts whitespace to some reasonable degree.
        return parseInt(event.target.value);
    },
    initialize: function() {
        this.model.on('change:distance', _.bind(this.render, this));
    },
    noKeypress: function(event) {
        if (event.keyCode === 13) {
            event.preventDefault();
            return false;
        }
    },
    noSubmit: function(event) {
        event.preventDefault();
        return false;
    },
    render: function() {
        this.$el.html(this.template(this.model.attributes));
        this.$el.find("button[name='distance']").on('click', _.bind(this.clickDistanceButton, this));
        this.$el.find('#vt-relations-parameters-form').on('keypress', _.bind(this.noKeypress, this));
        // Note: Although I haven't observed operating form elements to cause
        // the form to be submitted, this is a safety measure.
        this.$el.find('#vt-relations-parameters-form').on('submit', _.bind(this.noSubmit, this));
        this.$el.find('#vt-distance-field').on('keydown', _.bind(this.changeDistanceKeyDown, this));
        this.$el.find('#vt-distance-field').on('keyup', _.bind(this.changeDistanceKeyUp, this));
        this.model.bind('change:distance', _.bind(this.updateDistanceButtons, this));
        return this;
    },
    setDistance: function(value) {
        var d = parseInt(value);
        if (_.isEmpty(value)) {
            this.model.set('distance', '');
            this.$el.find('#vt-distance-field').removeClass('vt-validation-error');
            VT.Fn.trigger(VT.Components.FETCH_IRG_EVENT);
        } else if (!isNaN(d)) {
            this.model.set('distance', String(d));
            this.$el.find('#vt-distance-field').removeClass('vt-validation-error');
            VT.Fn.trigger(VT.Components.FETCH_IRG_EVENT);
        } else {
            // TODO Adding this class must result in change in visual appearance
            this.$el.find('#vt-distance-field').addClass('vt-validation-error');
        }
    },
    template: VT.Components.Templates.distanceInputGroup,
    updateDistanceButtons: function() {
        var v = this.model.get('distance');
        this.$el.find('button[name="distance"]').removeAttr('checked');
        this.$el.find(this.valueToDistanceButtonId(v)).attr('checked', 'checked');
    },
    updateDistanceField: function() {
        return AJS.$('#vt-distance-field')
            .val(this.model.get('distance'));
    },
    valueToDistanceButtonId: function(value) {
        if (_.indexOf(['0', '1', '2', '3'], value) >= 0) {
            return '#vt-distance-' + value;
        } else if (_.isEmpty(value)) {
            return '#vt-distance-unlimited';
        }
    }
});

VT.Components.GraphDirectionView = VT.Backbone.View.extend({
    click: function(event) {
        var v = event.target.value;
        this.model.set({graphDirection: v});
        VT.Fn.trigger(VT.Components.REDRAW_IRG_EVENT);
        return false;
    },
    initialize: function() {
        this.model.on('change:graphDirection', _.bind(this.render, this));
    },
    render: function() {
        this.$el.html(this.template(this.model.attributes));
        this.$el.find("input[name='graphDirection']").on('click', _.bind(this.click, this));
        return this;
    },
    template: VT.Components.Templates.graphDirectionInputGroup
});

VT.Components.ShowRelationshipLabelsSettingView = VT.Backbone.View.extend({
    changeShowRelationshipLabels: function() {
        var newState = this.model.get('showRelationshipLabels') === 'outward' ? 'false' : 'outward'; // Invert

        // Update the model.
        this.model.set('showRelationshipLabels', newState);
        VT.Fn.trigger(VT.Components.REDRAW_IRG_EVENT);
    },
    initialize: function() {
        this.model.on('change:showRelationshipLabels', _.bind(this.render, this));
    },
    render: function() {
        this.$el.html(this.options.template(this.model.toJSON()));
        this.$el.find('#vt-show-relationship-labels')
            .off('click.vt')
            .on('click.vt', _.bind(this.changeShowRelationshipLabels, this));
    }
});

VT.Components.IssueRelationGraphModel = VT.Backbone.Model.extend({
    getRelationsSeedIssuesArg: function() {
        var arg = this.get('relationsSeedIssuesArg');

        if (_.size(arg.trim()) <= 0) {
            return '';
        }

        // Try to match a list of issue keys.
        var issueKeys = VT.Components.asIssueKeys(arg);
        if (_.size(issueKeys) >= 1) {
            return '"jql = issueKey in (' + issueKeys.join(', ') + ')"';
        }

        // Otherwise, treat arg as a JQL query.
        return '"jql = ' + arg.replace(/"/g, '\\"') + '"';
    }
});

VT.Components.isCurrentRequestContext = function(model, requestContext) {
    var mrc = model.get('requestContext');
    return VT.Fn.get(mrc, 'requestUuid') == requestContext.requestUuid;
};

VT.Components.IssueRelationGraphView = VT.Backbone.View.extend({
    initialize: function() {
        _.bindAll(this, 'fetchGraphData', 'redrawGraph', 'resizeViewport');

        VT.Fn.bind(VT.Components.FETCH_IRG_EVENT, this.fetchGraphData);
        VT.Fn.bind(VT.Components.REDRAW_IRG_EVENT, this.redrawGraph);

        this.render();
    },
    fetchGraphData: function() {
        // Key is set only after the graph has been drawn once, so that the first draw
        // will not cause the display to indicate 'pending'.
        if (AJS.Meta.getBoolean('vt-use-pending')) { // TODO Investigate applying a data attribute to the container element instead. Necessary as the Confluence macro can be inserted several times in one page.
            this.$el.addClass('vt-pending');
        }

        // Prevent out-of-order display of graph information.
        // Without tracking requests, it's possible for replies to prior requests (perhaps longer running)
        // to arrive after the most recent request. The UI must always reflect the results of the most recent request.
        var requestContext = {
            requestUuid: VT.Fn.uuid.generate()
        };

        this.options.fetchGraphDataDelegate(requestContext);
    },
    redrawGraph: function() {
        var data = this.model.attributes;

        this.renderMessages();

        if (!VT.Fn.hasErrorMessages(data.messages)) {
            this.gfx = VT.Graph.draw(
                data,
                {
                    mountPointSel: '#vt-graph-container',
                    sortKeyForIssuesFn: VT.Trace.sortKeyForIssues,
                    viewportDimensionsHandler: VT.Trace.traceStudioViewportDimensions,
                    viewportModeFn: vivid.trace.jira.contextual_graph.data.viewport_mode_fn__js,
                }
            );
            // viewport-mode CLJS integration points:
            // A) Every time the relation graph is drawn.
            vivid.trace.jira.trace_studio.data.apply_viewport_mode__js(
                this.gfx,
                vivid.trace.jira.contextual_graph.data.viewport_mode_fn__js(),
            );
        }

        // Remove the pending indicator
        AJS.Meta.set('vt-use-pending', true);
        this.$el.removeClass('vt-pending');
    },
    render: function() {
        this.$el.html(this.template());
        this.renderMessages();

        if (this.options.renderShell) {
            this.options.renderShell(this);
        }

        this.renderLicenseBanner();
    },
    renderLicenseBanner: function() {
        var licenseBanner = this.model.get('licenseBanner');
        if (licenseBanner) {
            AJS.$('#vt-license-banner-container').html(
                VT.Components.Templates.licenseBanner({
                    licenseBanner: licenseBanner
                })
            );
        }
    },
    renderMessages: function() {
        this.$el.find('#vt-message-container').empty();

        var _this = this;
        function msg(messages, sel, templateFn) {
            if (_.size(messages) >= 1) {
                _this.$el.find(sel).append(
                    templateFn({
                        messages: _.map(messages, function(v, code) {
                            return VT.Fn.renderMessage(code, v);
                        })
                    })
                );
            }
        }

        var messages = _.omit(
            this.model.get('messages'),
            'VTW-5', // TRACE-579: Warning VTW-5 is unnecessary within the context of the Trace Studio.
            'VTW-16' // Warning VTW-16 is appropriate for JQL functions, but might be intentional in project configurations. TODO add VTW-16's originating message to the message reporter interface, and eliminate this special casing.
        );

        var errorMessages = VT.Fn.allErrorMessages(messages);
        msg(errorMessages, '#vt-message-container', VT.Components.Templates.messageErrors);
        if (_.size(errorMessages) >= 1) {
            // Remove the graph
            this.$el.find('#vt-graph-container').empty();
        }

        var warningMessages = VT.Fn.allWarningMessages(messages);
        msg(warningMessages, '#vt-message-container', VT.Components.Templates.messageWarnings);
    },
    resizeViewport: function(opts) {
        var containerEl = AJS.$('#vt-graph-container');
        if (opts.sizeCSS && opts.viewportWidth) {
            containerEl.css('width', opts.viewportWidth);
        }
        if (opts.sizeCSS && opts.viewportHeight) {
            containerEl.css('height', opts.viewportHeight);
        }

        if (this.gfx) {
            this.gfx.viewportWidth = opts.viewportWidth;
            this.gfx.viewportHeight = opts.viewportHeight;
            VT.Graph.resize(this.gfx);
        }
    },
    template: VT.Components.Templates.issueRelationGraph
});

VT.Components.renderHelp = function(selectorFragment, helpTemplate) {
    var el = '#vt-' + selectorFragment + '-help';
    AJS.$(el).html(
        helpTemplate()
    );
};

// Largely copy & paste of VT.Components.renderIssueLinkTypeEditor.
// Dangerous, perhaps, but I intend for this code to be obliterated during the CLJS rewrite.
VT.Components.renderArtifactTypeEditor = function(sel) {
    vivid.trace.jira.trace_studio.core.artifact_types_view__js(sel);
};

VT.Components.renderIssueLinkTypeEditor = function(sel) {
    vivid.trace.jira.trace_studio.core.issue_link_types_view__js(sel);
};

VT.Components.renderItemCardLayoutEditor = function(sel) {
    vivid.trace.jira.trace_studio.core.item_card_editor_view__js(sel);
};

VT.Components.wireBulkChangeButton = function(el, jqlQuery) {
    AJS.$(el)
        .attr('href',
            VT.Fn.applicationBaseUrl() +
            '/plugins/servlet/vivid.trace/bulk-operation?jqlQuery=' + encodeURIComponent(jqlQuery)
        );
};

VT.Components.wireButtonClick = function(el, fn) {
    AJS.$(el)
        .off('click.vt')
        .on('click.vt', fn);
};

VT.Components.wireListIssuesButton = function(el, jqlQuery) {
    AJS.$(el)
        .attr('href',
            VT.Fn.applicationBaseUrl() +
            '/secure/IssueNavigator.jspa?reset=true&jqlQuery=' + encodeURIComponent(jqlQuery)
        );
};
