/**
 * @license
 * Copyright 2025 Vivid Inc. and/or its affiliates.
 */

"use strict";

// TODO Tweak edge arrowhead abutment
// TODO What happens if an issue is related to itself? How is the edge drawn?
// TODO Deal with situations where the user has since been logged out on a screen,
//  and the user does an operation which requires them to be logged in, such as "Configure Project".

/*
 * Algorithms convert the essential graph data into a display list.
 * Each entry describes a subgraph, in the case of the last element, optionally the orphans.
 *
 *   display: [
 *     { height: 96
 *       sortkey: "ABC-1"
 *       draw_fn: <draw_subgraph_fn>
 *       layout_data: <data> }
 *     { height: 64
 *       sortkey: "ABC-7"
 *       draw_fn: <draw_subgraph_fn>
 *       layout_data: <data> }
 *     ...
 *     { height: 32
 *       draw_fn: <draw_orphans_fn>
 *       layout_data: [ issue keys ... ] } ]
 *
 *
 *   D R A W I N G
 *
 * Align x & y to integral boundaries.
 *
 * All elements appearing in the Issue Relation Graph, including text, must be drawn within
 * the SVG if they are to be included in exported files.
 */

if (typeof VT == 'undefined') {
    var VT = {};
}
if (typeof VT.Graph == 'undefined') {
    VT.Graph = {};
}

/**
 * Signals for the removal of the hover spotlight in the graph.
 */
VT.Graph.REMOVE_SPOTLIGHT_EVENT = 'VividTraceGraphRemoveSpotlight';

/**
 * ADG Colors
 * Referring to https://design.atlassian.com/latest/product/foundations/colors/
 */
VT.ADG_Colors = {
    // Atlassian AUI Color: B100 Arvo breeze
    Blue: '#4c9aff',
    LightGray: '#f5f5f5',
    MediumGray: '#707070'
};

/*
 * Collapses the relations between nodes into paired entries that map one or more relation between either node.
 *   Node A  ->  node B  -> {   // Tuple (Node A, node B)
 *     relations: [ Array of relation triplets, all specific to nodes A & B only ]
 *     arrowheads: { Issue key of Node A and/or B: true }
 *   }
 */
VT.Graph.calculateCollapsedEdges = function(edges) {
    var c = {};
    var p = function(edge, c) {
        var v = edge[0],
            relation = edge[1],
            w = edge[2];
        if (typeof c[v] !== 'undefined' && typeof c[v][w] !== 'undefined') {
            var entry = c[v][w];
            entry.relations.push([v, relation, w]);
            entry.arrowheads[w] = true;
        } else if (typeof c[w] !== 'undefined' && typeof c[w][v] !== 'undefined') {
            var entry = c[w][v];
            entry.relations.push([v, relation, w]);
            entry.arrowheads[w] = true;
        } else {
            if (typeof c[v] === 'undefined') {
                c[v] = {};
            }
            c[v][w] = {
                relations: [[v, relation, w]],
                arrowheads: {}
            };
            c[v][w].arrowheads[w] = true;
        }
    };

    _.each(edges, function (edge) {
        p(edge, c);
    });

    return c;
};

VT.Graph.calculateEdgeCoordinates = function(data, gfx, layout_data, edge) {
    // Use a margin between the arrowhead tip and issue lozenge border to allow the arrowheads to
    // remain visible, even when the edge is at its angular extreme.
    var PX_MARGIN = 1;

    // Calculate center points of line segments
    function left(node) {
        return {x: node.x - PX_MARGIN, y: node.y + node.height / 2};
    }
    function right(node) {
        return {x: node.x + PX_MARGIN + node.width, y: node.y + node.height / 2};
    }
    function top(node) {
        return {x: node.x + node.width / 2, y: node.y - PX_MARGIN};
    }
    function bottom(node) {
        return {x: node.x + node.width / 2, y: node.y + PX_MARGIN + node.height};
    }

    var PI_2 = Math.PI / 2;
    function angleOf(a, b, phase) {
        var angle = Math.atan2(b.y - a.y, b.x - a.x) - phase;
        var sign = angle < 0 ? -1 : 1;
        if (Math.abs(angle) >= PI_2) {
            angle = sign * (Math.PI - Math.abs(angle));
        }

        // Scale number from [-PI, PI] to [-1, 1]
        angle /= Math.PI;
        return angle;
    }

    var isHorizontal = VT.Graph.isGraphDirectionHorizontal(data);

    // Graphic representation: To draw the edge with its ends touching exactly upon issue lonzenge's curved edges
    // requires determining where the line intersects the issue lozenge's curved edge; this involves the
    // computationally-expensive measure of solving a quadratic equation with complicated coefficients.
    //
    // So as the computationally less-expensive option, we instead calculate the abutment point along the
    // straightened edge segment as a function of the edge's angle.

    var v = layout_data.node(edge.v);
    var w = layout_data.node(edge.w);
    var a, b, xOffset = 0, yOffset = 0;
    if (isHorizontal) {
        // Horizontally, is v to the left of w?
        var is_v_left_of_w = v.x < w.x;
        a = is_v_left_of_w ? right(v) : left(v);
        b = is_v_left_of_w ? left(w) : right(w);
        // Tighten the range of edge abutment to maintain clarity.
        yOffset = angleOf(a, b, 0) * (gfx.NODE_HEIGHT - gfx.EM);
    } else {
        // Vertically, is v above w?
        var is_v_above_w = v.y < w.y;
        a = is_v_above_w ? bottom(v) : top(v);
        b = is_v_above_w ? top(w) : bottom(w);
        // Tighten the range of edge abutment to maintain clarity.
        xOffset = angleOf(a, b, PI_2) * (gfx.NODE_WIDTH / 2);
    }

    return { coords: {
        a: {x: a.x - xOffset, y: a.y + yOffset},
        b: {x: b.x + xOffset, y: b.y - yOffset}
    }};
};

VT.Graph.computeDisplayList = function(graphs, gfx, data) {
    // Layout subgraphs and (optionally) consolidate orphans.
    var orphans = [];
    var subgraphs = [];
    _.each(graphs, function(graph) {
        if (data.labelOrphans && graph.relations.length === 0) {
            orphans.push(graph.issues[0]);
        } else {
            var entry = VT.Graph.layoutSubgraph(graph, gfx, data);
            subgraphs.push(entry);
        }
    });

    // Produce display list.
    var displayList = subgraphs.sort(function(a, b) {
        return VT.Fn.issueKeyComparator(a.sortkey, b.sortkey);
    });

    if (orphans.length >= 1) {
        displayList.push(VT.Graph.layoutOrphans(orphans, gfx, data));
    }

    return displayList;
};

VT.Graph.computeMargins = function(data, gfx) {
    var isShowLabels = data.showRelationshipLabels === 'outward';
    var isHorizontal = VT.Graph.isGraphDirectionHorizontal(data);
    var maxRelationshipLabelDimensions = VT.Graph.computeMaxRelationshipLabelDimensions(data, gfx);
    return isHorizontal ? {
        MARGIN: 10,
        H_MARGIN: isShowLabels ? maxRelationshipLabelDimensions.width + 60 : 40,
        V_MARGIN: 20
    } : {
        MARGIN: 10,
        H_MARGIN: 20,
        V_MARGIN: isShowLabels ? maxRelationshipLabelDimensions.height + 40 : 40
    };
};

VT.Graph.computeMaxRelationshipLabelDimensions = function(data, gfx) {
    // Run through all labels seeking the widest pixel width and tallest pixel height.
    var text = VT.Graph.SVG.textNull(gfx);
    var dim = _.reduce(data.relations, function(memo, r) {
        var labelText = r[1];
        text.text(labelText);
        var bbox = VT.Graph.SVG.bbox(text);
        return {
            width: Math.ceil(Math.max(memo.width, bbox.width)),
            height: Math.ceil(Math.max(memo.height, bbox.height))
        };
    }, {width: 0, height: 0});
    text.remove();
    return dim;
};

VT.Graph.cursorOnCurrentIssue = function(currentIssueKey, gfx) {
    var elementId = '#' + this.elementIdForNode(currentIssueKey);
    var issue = VT.Graph.SVG.getElementsByCSSSelector(gfx, elementId)[0];
    if (!_.isUndefined(issue)) {
        var cursor = VT.Graph.SVG.issueCursor(gfx, issue);
        gfx.layers.backgroundLayer.add(cursor);
        return cursor;
    }
};

VT.Graph.dagreRankDirForGraphDirection = function(graphDirection) {
    return VT.Graph.GRAPH_DIRECTION_TO_DAGRE_RANK_DIR[graphDirection] ||
        VT.Graph.GRAPH_DIRECTION_TO_DAGRE_RANK_DIR.right;
};

VT.Graph.defaultIssueFieldHandler = function(id) {
    return {
        draw: function(data, gfx, issue, x, y, width, height) {
            var el;
            if (_.isObject(issue[id])) {
                switch (issue[id].type) {
                    case 'date':
                        el = VT.Graph.drawDate(data, gfx, x, y, width, issue[id].value);
                        break;
                    case 'datetime':
                        el = VT.Graph.drawDate(data, gfx, x, y, width, issue[id].value, true);
                        break;
                    case 'url':
                        el = VT.Graph.SVG.link(data, gfx, issue[id].value);
                        var t = VT.Graph.SVG.text(gfx, x, y, issue[id].value, width, height);
                        VT.Graph.SVG.stylizeHyperlink(gfx, t);
                        el.add(t);
                        break;
                    default:
                        el = VT.Graph.SVG.text(gfx, x, y, issue[id].value, width, height);
                }
            } else {
                el = VT.Graph.SVG.text(gfx, x, y, issue[id], width, height);
            }
            return el;
        }
    };
};

/*
 * Creates a mapping of subgraph IDs to issues in that subgraph. Each partition of related issues, a subgraph, is assigned its
 * own subgraph ID. Some subgraphs may have only one issue and no relations; these issues are graphed as orphans.
 *
 *  [ { relations: [ ["ABC-1", "Blocks", "ABC-2"], ... ],
 *      issues: ["ABC-1", "ABC-2", ..., "ABC-15"] },
 *    { relations: [ ["ABC-81", "Cloners", "ABC-79"] ],
 *      issues: ["ABC-79", "ABC-81"] },
 *    ...
 *    { relations: [],
 *      issues: ["ABC-99"] },
 *  ]
 */
VT.Graph.divideIntoSubgraphs = function(issues, relations) {
    var subgraphId = (function() {
        var i = 0;
        return function() {
            return i += 1;
        };
    })();
    var i2g = {};
    var g2r = {};

    var record = function(relation) {
        var l = relation[0];
        var r = relation[2];

        if ((l in i2g) && (r in i2g)) {
            var srcg = i2g[r];
            var destg = i2g[l];
            g2r[destg].relations.push(relation);
            if (srcg != destg) {
                // Merge subgraphs
                var src = g2r[i2g[r]];
                _.each(src.issues, function(i) {
                    i2g[i] = destg;
                    g2r[destg].issues.push(i);
                });
                _.each(src.relations, function(srcr) {
                    g2r[destg].relations.push(srcr);
                });
                delete g2r[srcg];
            }
        } else if (!(l in i2g) && !(r in i2g)) {
            // New subgraph
            g = subgraphId();
            i2g[l] = g;
            i2g[r] = g;
            g2r[g] = { relations: [relation], issues: [l, r] };
        } else {
            // Add node to existing subgraph
            var g = undefined;
            if (l in i2g) {
                g = i2g[l];
                i2g[r] = g;
                g2r[g].issues.push(r);
            } else {
                g = i2g[r];
                i2g[l] = g;
                g2r[g].issues.push(l);
            }
            g2r[g].relations.push(relation);
        }
    };
    _.each(relations, function(relation) { record(relation); });

    // Add solitary issues that will become orphans
    _.each(issues, function(i) {
        var key = i[':jira.issue-field.system/key'];
        if (!(key in i2g)) {
            g2r[subgraphId()] = { relations: [], issues: [key]};
        }
    });
    return g2r;
};

/*
 * Design rationale:
 *
 * We must factor in the earliest/oldest supported browser versions as the lowest common denominator when considering
 * technology decisions that affect the project's goals, particularly leveraging modern web standards such as HTML 5,
 * CSS 3, and modern JavaScript technologies to reduce development costs.
 *
 * For graphing, we use svj.js, a lightweight library for manipulating SVG.
 *
 * Subgraphs are vertically stacked and are separated by a double-wide margin, so that distinguishing between
 * subgraph outlines is easier for readers.
 *
 * Issues with no relations are corralled at the bottom of the graph, and are labelled as "Orphans."
 * A narrow margin between issues yields a denser cluster of issues, and visually distinguishes the cluster of
 * orphans from other issue arrangements in the graph. Sorted alphabetically, a person can scan in a natural
 * reading direction for issues of interest.
 */
VT.Graph.draw = function(data, viewOptions) {
    var containerEl = VT.Fn.clearElement(viewOptions.mountPointSel);

    data.sortKeyForIssuesFn = viewOptions.sortKeyForIssuesFn;

    // A portion of the relations might involve nodes for which we know nothing about; discard them on the client-side
    // to reduce server-side processing load. Excess or incomplete relation data can be due results being truncated to
    // the Issue Count Soft Maximum, or disabled issue link types.
    var issueKeys = _.keys(data.issues);
    data.relations = _.reject(data.relations, function(r) {
        return !(_.contains(issueKeys, r[0])) || !(_.contains(issueKeys, r[2]));
    });

    // Layout and draw the Issue Relation Graph.
    var gfx = {
        mountPointSel: viewOptions.mountPointSel,
        EM: 14,
        em: function(ems) {
            return this.EM * ems;
        },
        paper: VT.Graph.SVG.paper(AJS.$(viewOptions.mountPointSel)[0]),
        viewportDimensionsHandler: viewOptions.viewportDimensionsHandler,
        viewportModeFn: viewOptions.viewportModeFn,
        theme: VT.Theme,
    };

    _.extend(
        gfx,
        VT.Graph.computeMargins(data, gfx)
    );

    _.extend(
        gfx,
        vivid.trace.lib.views.relation_graph.calculate_node_dimensions__js(data.itemCardLayout)
    );

    gfx.resize = function() {
        VT.Graph.resize(gfx);
    };

    // Index partitions of issues that form their own subgraphs by ID.
    var subgraphs = VT.Graph.divideIntoSubgraphs(data.issues, data.relations);

    // Layout each subgraph, consolidating orphans into their own pool, and produce a display list.
    var displayList = VT.Graph.computeDisplayList(subgraphs, gfx, data);

    VT.Graph.drawDisplayList(displayList, data, gfx);

    _.extend(
        gfx,
        viewOptions.viewportDimensionsHandler(gfx.displayListWidth, gfx.displayListHeight)
    );

    // Resize once, now
    VT.Graph.resize(gfx);

    // TODO Suitable for :scroll view-mode. For :zoom mode, use a different mechanism.
    // Place the current issue in the center of the view
    if (data.currentIssueKey) {
        VT.Graph.scrollViewToIssue(data.currentIssueKey, gfx);
    }

    VT.Fn.bind(
        VT.Graph.REMOVE_SPOTLIGHT_EVENT,
        function() {
            VT.Graph.spotlightRemove(gfx);
        }
    );

    return gfx;
};

VT.Graph.drawDate = function(data, gfx, x, y, width, text, completeDate) {
    var userLocale = AJS.$('meta[name="ajs-user-locale"]').attr("content");
    var dateString = _vt_moment(parseInt(text))
            .utcOffset(data.userTimeZone)
            .locale(userLocale)
            .strftime(completeDate ? data.dateTimePickerJavascriptFormat : data.datePickerJavascriptFormat)
        ;

    return VT.Graph.SVG.text(gfx, x, y, dateString, width);
};

VT.Graph.drawDisplayList = function(displayList, data, gfx) {
    // Use the horizontal margin to compute the minimum width of the paper
    var displayListWidth = _.reduce(displayList, function(memo, entry) {
            return Math.max(memo, entry.widthPx);
        }, 0) + gfx.MARGIN;
    gfx.displayListWidth = displayListWidth;

    // Calculate the minimum height of the paper.
    if (_.size(displayList) >= 1) {
        var displayListHeight = _.chain(displayList)
            .map(function (o) {
                return o.height;
            })
            .reduce(function (o, memo) {
                return o + memo;
            }, 0)
            .value();
        gfx.displayListHeight = displayListHeight;
    } else {
        // No issues:
        gfx.displayListHeight = 0;
    }

    // Initialize SVG defines
    VT.Graph.SVG.initializeDefs(gfx);

    // Set up layers
    gfx.layers = {
        backgroundLayer: VT.Graph.SVG.set(gfx),
        edgeLayer: VT.Graph.SVG.set(gfx),
        issueLayer: VT.Graph.SVG.set(gfx),
        labelLayer: VT.Graph.SVG.set(gfx)
    };

    if (displayList.length >= 1) {
        var y_offset = 0;
        _.each(displayList, function (o) {
            o.draw_fn(o.layout_data, data, gfx, y_offset, o.height);
            y_offset += o.height;
        });
    } else {
        // No issues:
        var label = VT.Graph.drawLabel(
            data,
            gfx,
            gfx.MARGIN,
            gfx.MARGIN,
            AJS.I18n.getText('vivid.phrase.empty')
        );

        gfx.displayListWidth = 0;
    }

    if (data.currentIssueKey) {
        this.cursorOnCurrentIssue(data.currentIssueKey, gfx);
    }

    // Wallpaper
    gfx.wallpaper = VT.Graph.SVG.wallpaper(gfx);
    gfx.layers.backgroundLayer.add(gfx.wallpaper);

    // Order the layers once, now, to correctly order viewing display.
    VT.Graph.orderLayers(gfx);
};

VT.Graph.addCommonStylesheetToElementNode = function(el) {
    const style = document.createElementNS("http://www.w3.org/2000/svg", 'style');
    style.innerHTML = ".unspotlit { opacity: 0.4; transition: opacity 0.1s; transition-delay: 0.5s; }"
    el.node.appendChild(style);
}

VT.Graph.drawIssue = function(data, gfx, issue, x, y) {
    var g = VT.Graph.SVG.group(gfx);
    VT.Graph.addCommonStylesheetToElementNode(g);
    VT.Graph.SVG.assignElementId(g, VT.Graph.elementIdForNode(issue[':jira.issue-field.system/key']));
    gfx.layers.issueLayer.add(g);

    // Some parts of the issue lozenge are intentionally hyperlinked (grouped by "specific <a>"s); the rest need
    // to be hyperlinked to jump to the given issue so that the issue lozenge in its entirety is
    // clickable (the "general <a>").
    // The SVG specification forbids nested <a> elements, so any specific <a> element children need to be placed:
    // - After the general <a>, so that they will be drawn above the issue lozenge background, and
    // - As peers to the general <a> to avoid nesting.

    var a = VT.Graph.setHyperlinkToIssue(gfx, issue[':jira.issue-field.system/key'], data);
    g.add(a);

    // Draw the issue lozenge
    var rect = VT.Graph.SVG.issueLozenge(gfx, x, y, issue);
    a.add(rect);

    // Feature: An issue's key and summary can be discovered by inspecting
    // the issue lozenge's tooltip.
    VT.Graph.tooltipTextForIssue(gfx, issue, rect);

    function addElToUnit(el) {
        if (el.node.nodeName === 'a') {
            g.add(el);
        } else {
            VT.Graph.SVG.addTo(a, el);
        }
    }

    // Draw item card fields
    var itemCardLayout = vivid.trace.jira.trace_studio.core.edn_to_js_obj__js(data.itemCardLayout);
    var x_offset = gfx.em(0.95);
    var y_offset = gfx.em(0.45);
    _.each(itemCardLayout, function(field) {
        if (issue[field.id] !== undefined) {
            var x_px = vivid.trace.lib.views.relation_graph.unit_coordinate_in_px(field.x);
            var y_px = vivid.trace.lib.views.relation_graph.unit_coordinate_in_px(field.y);
            var width_px = vivid.trace.lib.views.relation_graph.unit_dimension_in_px(field.w);
            var height_px = vivid.trace.lib.views.relation_graph.unit_dimension_in_px(field.h);
            var handler = VT.Graph.issueFieldHandler(field.id);
            var els = handler.draw(
                data,
                gfx,
                issue,
                // Draw on pixel boundaries
                Math.floor(x + x_px + x_offset),
                Math.floor(y + y_px + y_offset),
                Math.ceil(width_px),
                Math.ceil(height_px)
            );

            // Specific <a> elements need to peered with the general <a>;
            // everything else can be appended to the general <a>.
            if (_.isArray(els)) {
                _.each(els, addElToUnit);
            } else {
                addElToUnit(els);
            }
        }
    });

    return g;
};

VT.Graph.drawLabel = function(data, gfx, x, y, str) {
    var text = VT.Graph.SVG.text(gfx, x, y, str);
    gfx.layers.labelLayer.add(text);
    return text;
};

VT.Graph.drawOrphans = function(layout_data, data, gfx, y_offset, height) {
    var x = gfx.MARGIN, y = y_offset + gfx.MARGIN;

    if (data.labelOrphans) {
        var margin = gfx.em(0.5);
        var text_y_offset = gfx.em(0.5);
        VT.Graph.drawLabel(
            data,
            gfx,
            x,
            y + margin - text_y_offset,
            AJS.I18n.getText("vivid.trace.issue-relation-graph.orphan-issues-label")
        );
        y += gfx.em(1) + gfx.MARGIN;
    }

    var _this = this;
    function setEventHandlersOn(centric_fn, el) {
        el.on('mouseenter', _.bind(VT.Graph.spotlight, _this, centric_fn, gfx));
        el.on('mouseleave', _.partial(VT.Graph.spotlightRemove, gfx));
    }

    var stride = gfx.NODE_WIDTH + gfx.MARGIN;
    var idx = 0;
    _.each(layout_data.issueKeys, function(issueKey) {
        var issue = data.issues[issueKey];
        var g = VT.Graph.drawIssue(data, gfx, issue, x, y);

        // Feature: Hovering over an orphan issue spotlights the issue.
        setEventHandlersOn(function() { return [VT.Graph.elementIdForNode(issue[':jira.issue-field.system/key'])]; }, g);

        x += stride;
        idx++;
        if ((idx % layout_data.issuesPerLine) === 0) {
            x = gfx.MARGIN;
            y += gfx.NODE_HEIGHT + gfx.MARGIN;
        }
    });
};

VT.Graph.drawRelation = function(data, gfx, layout_data, edge, y_offset) {
    var g = VT.Graph.SVG.group(gfx);
    VT.Graph.SVG.assignElementId(g, VT.Graph.elementIdForEdge(edge));

    var edgeEl = VT.Graph.SVG.relationEdge(gfx, layout_data, edge, y_offset);
    VT.Graph.SVG.addTo(g, edgeEl);
    VT.Graph.SVG.addTo(gfx.layers.edgeLayer, edgeEl);

    return g;
};

VT.Graph.drawRelationLabel = function(gfx, layout_data, edge, y_offset) {
    // Calculate coordinates
    var src = edge.v,
        dst = edge.w;
    var collapsedEdge = layout_data.edge(src, dst).collapsedEdge;
    var relationsCount = collapsedEdge.relations.length;
    var firstRelation = collapsedEdge.relations[0];
    var relationLabel = relationsCount >= 2 ? firstRelation[1] + ' +' + (relationsCount - 1) : firstRelation[1];
    var x_mid = (edge.coords.a.x + edge.coords.b.x) / 2;
    var y_mid = (edge.coords.a.y + edge.coords.b.y) / 2 + y_offset - (gfx.theme.FontSizeInPx / 2);

    // Feature: A full list of relations is available in the label tooltip.
    var relationList = _.map(collapsedEdge.relations, function(r) {
        return r[0] + ' ' + r[1] + ' ' + r[2];
    }).join("\n");

    var el = VT.Graph.SVG.relationLabel(gfx, relationLabel, relationList, x_mid, y_mid);
    VT.Graph.SVG.assignElementId(el, VT.Graph.elementIdForEdgeLabel(edge));
    return el;
};

VT.Graph.drawSubgraph = function(layout_data, data, gfx, y_offset, height) {
    var _this = this;
    function setEventHandlersOn(centric_fn, el) {
        el.on('mouseenter', _.bind(VT.Graph.spotlight, _this, centric_fn, gfx));
        el.on('mouseleave', _.partial(VT.Graph.spotlightRemove, gfx));
    }

    // Jira issues are represented as vertices
    _.each(layout_data.nodes(), function(node) {
        var info = layout_data.node(node);
        var issue = data.issues[node];
        var g = VT.Graph.drawIssue(data, gfx, issue, info.x, info.y + y_offset);

        // Feature: Hovering over an issue spotlights the issue and all immediate relations.
        setEventHandlersOn(_.partial(VT.Graph.spotlightIssueCentric, node, layout_data), g);
    });

    // Relations between Jira issues are represented as edges (lines between pairs of issues)
    var isShowLabels = data.showRelationshipLabels === 'outward';
    _.each(layout_data.edges(), function(edge) {
        var s = VT.Graph.SVG.set(gfx);

        _.extend(edge, VT.Graph.calculateEdgeCoordinates(data, gfx, layout_data, edge));
        var edgeEl = VT.Graph.drawRelation(data, gfx, layout_data, edge, y_offset);
        VT.Graph.SVG.addTo(s, edgeEl);

        if (isShowLabels) {
            var labelEl = VT.Graph.drawRelationLabel(gfx, layout_data, edge, y_offset);
            VT.Graph.SVG.addTo(gfx.layers.labelLayer, labelEl);
            VT.Graph.SVG.addTo(s, labelEl);
        }

        // Feature: Hovering over a relation spotlights the relation and related issues.
        setEventHandlersOn(_.partial(VT.Graph.spotlightRelationCentric, edge), s);
    });
};

VT.Graph.drawUser = function(gfx, x, y, width, user) {
    return VT.Graph.SVG.text(gfx, x, y, user.displayName, width);
//    var url = VT.Fn.ajsContextPath() + '/secure/ViewProfile.jspa?name=' + user.name;
};

/**
 * Structures HTML element IDs for SVG components.
 */
VT.Graph.elementId = function(type, id) {
    return 'vt-' + type + '--' + id;
};
VT.Graph.elementIdStem = function(type) {
    return 'vt-' + type + '--';
};

VT.Graph.elementIdForEdge = function(edge) {
    return VT.Graph.elementId('relation', edge.v + '-' + edge.w);
};
VT.Graph.elementIdForEdgeStem = function() {
    return VT.Graph.elementIdStem('relation');
};

VT.Graph.elementIdForEdgeLabel = function(edge) {
    return VT.Graph.elementId('relation-label', edge.v + '-' + edge.w);
};
VT.Graph.elementIdForEdgeLabelStem = function() {
    return VT.Graph.elementIdStem('relation-label');
};

VT.Graph.elementIdForNode = function(node) {
    return VT.Graph.elementId('issue', node);
};
VT.Graph.elementIdForNodeStem = function() {
    return VT.Graph.elementIdStem('issue');
};

VT.Graph.GRAPH_DIRECTION_TO_DAGRE_RANK_DIR = {
    down: 'TB',
    left: 'RL',
    right: 'LR'
};

/*
 * Feature: On hover (i.e. mouseover) spotlight the item and its immediate relations.
 * The spotlight is effected by reducing the opacity of all other items in the graph.
 * These elements are "unspotlit" as in "not spotlit".
 */
VT.Graph.spotlight = function (centric_fn, gfx) {
    if (VT.Graph.spotlightIsActive(gfx)) {
        return;
    }

    // Add unspotlit class to all issues and relations in the graph not identified by the centric fn.
    const ids      = centric_fn();
    const allParts = document.querySelectorAll(
        "#vt-graph-container *[id^='" + VT.Graph.elementIdForNodeStem() +"']," +
            "#vt-graph-container *[id^='" + VT.Graph.elementIdForEdgeStem() +"']," +
            "#vt-graph-container *[id^='" + VT.Graph.elementIdForEdgeLabelStem() +"']"
    );
    _.each(allParts, function (el) {
        if (_.contains(ids, el.id)) {
            // TODO Order el to the front of its SVG layer.
        } else {
            el.classList.add(gfx.theme.UnspotlitCSSClassName)
        }
    })
};

VT.Graph.spotlightIsActive = function(gfx) {
    const els = VT.Graph.SVG.getElementsByCSSSelector(gfx, '.' + gfx.theme.UnspotlitCSSClassName);
    return _.size(els) >= 1;
};

VT.Graph.spotlightIssueCentric = function(node, layout_data) {
    function otherNode(edge) {
        var other = node === edge.v ? edge.w : edge.v;
        return VT.Graph.elementIdForNode(other);
    }
    var edges = layout_data.nodeEdges(node);
    // Arranged in increasing order of priority to align with SVG's "painter's order," so that
    // higher-priority elements obscure lower-priority ones.
    return _.union(
        _.map(edges, otherNode),
        _.map(edges, VT.Graph.elementIdForEdge),
        _.map(edges, VT.Graph.elementIdForEdgeLabel),
        [VT.Graph.elementIdForNode(node)]
    );
};

VT.Graph.spotlightRelationCentric = function(edge) {
    return [
        VT.Graph.elementIdForNode(edge.v),
        VT.Graph.elementIdForNode(edge.w),
        VT.Graph.elementIdForEdge(edge),
        VT.Graph.elementIdForEdgeLabel(edge)
    ];
};

VT.Graph.spotlightRemove = function(gfx) {
    const els = VT.Graph.SVG.getElementsByCSSSelector(gfx, '.' + gfx.theme.UnspotlitCSSClassName);
    _.each(els, function(el) {
        el.node.classList.remove(gfx.theme.UnspotlitCSSClassName);
    })
};

VT.Graph.isGraphDirectionHorizontal = function(data) {
    return ! _.contains(
        ['down'],
        (data.graphDirection || 'right').toLowerCase()
    );
};

VT.Graph.issueFieldHandler = function(id) {
    return id in VT.Graph.issueFieldHandlerMap ?
        VT.Graph.issueFieldHandlerMap[id] :
        VT.Graph.defaultIssueFieldHandler(id);
};

VT.Graph.issueFieldHandlerMap = {
    ':jira.issue-field.system/assignee': {
        draw: function(data, gfx, issue, x, y, width, height) {
            return VT.Graph.drawUser(gfx, x, y, width, data.userMap[issue[':jira.issue-field.system/assignee']]);
        }
    },
    ':jira.issue-field.system/created': {
        draw: function(data, gfx, issue, x, y, width, height) {
            return VT.Graph.drawDate(data, gfx, x, y, width, issue[':jira.issue-field.system/created']);
        }
    },
    ':jira.issue-field.system/due-date': {
        draw: function(data, gfx, issue, x, y, width, height) {
            return VT.Graph.drawDate(data, gfx, x, y, width, issue[':jira.issue-field.system/due-date']);
        }
    },
    ':jira.issue-field.system/issue-type': {
        draw: function(data, gfx, issue, x, y, width, height) {
            var issuetype = data.issueTypeMap[issue[':jira.issue-field.system/issue-type']];
            var url = VT.Fn.applicationBaseUrl() + issuetype.iconUrl;
            return VT.Graph.SVG.image(gfx, x, y, url, width, issuetype.name);
        }
    },
    ':jira.issue-field.system/key': {
        draw: function(data, gfx, issue, x, y, width, height) {
            var els = [];

            var el = VT.Graph.SVG.text(gfx, x, y, issue[':jira.issue-field.system/key'], width);
            els.push(el);

            VT.Graph.SVG.stylizeHyperlink(gfx, el);
            var isResolved = 'resolution' in issue;
            if (isResolved) {
                var strikethrough = VT.Graph.SVG.textStrikethrough(gfx, el, gfx.theme.LinkText);
                els.push(strikethrough);
            }
            VT.Graph.tooltipTextForIssue(gfx, issue, el);

            return els;
        }
    },
    ':jira.issue-field.system/priority': {
        draw: function(data, gfx, issue, x, y, width, height) {
            var priority = data.issuePriorityMap[issue[':jira.issue-field.system/priority']];
            var url = VT.Fn.applicationBaseUrl() + priority.iconUrl;
            return VT.Graph.SVG.image(gfx, x, y, url, width, priority.name);
        }
    },
    ':jira.issue-field.system/project': {
        draw: function(data, gfx, issue, x, y, width, height) {
            var project = data.projectMap[issue[':jira.issue-field.system/project']];
            //var url = VT.Fn.ajsContextPath() + '/browse/' + project.key + '#selectedTab=vivid.trace:project-trace-tabpanel';
            return VT.Graph.SVG.text(gfx, x, y, project.name, width);
        }
    },
    ':jira.issue-field.system/reporter': {
        draw: function(data, gfx, issue, x, y, width, height) {
            return VT.Graph.drawUser(gfx, x, y, width, data.userMap[issue[':jira.issue-field.system/reporter']]);
        }
    },
    ':jira.issue-field.system/resolution': {
        draw: function(data, gfx, issue, x, y, width, height) {
            var resolution = data.issueResolutionMap[issue[':jira.issue-field.system/resolution']];
            return VT.Graph.SVG.text(gfx, x, y, resolution, width);
        }
    },
    ':jira.issue-field.system/resolution-date': {
        draw: function(data, gfx, issue, x, y, width, height) {
            return VT.Graph.drawDate(data, gfx, x, y, width, issue[':jira.issue-field.system/resolution-date']);
        }
    },
    ':jira.issue-field.system/status': {
        draw: function(data, gfx, issue, x, y, width, height) {
            var status = data.issueStatusMap[issue[':jira.issue-field.system/status']];
            return VT.Graph.SVG.text(gfx, x, y, status, width);
        }
    },
    ':jira.issue-field.system/updated': {
        draw: function(data, gfx, issue, x, y, width, height) {
            return VT.Graph.drawDate(data, gfx, x, y, width, issue[':jira.issue-field.system/updated']);
        }
    }
};

VT.Graph.layoutOrphans = function(orphans, gfx, data) {
    var viewportDims = gfx.viewportDimensionsHandler();
    var issueDrawingSpacePerLine = (viewportDims.viewportWidth || gfx.displayListWidth) - (gfx.MARGIN * 2);
    var issuesPerLine = Math.max(1, 1 + Math.floor((issueDrawingSpacePerLine - gfx.NODE_WIDTH) / (gfx.MARGIN + gfx.NODE_WIDTH)));
    var lines = Math.ceil(orphans.length / issuesPerLine);
    var maxColumns = Math.min(orphans.length, issuesPerLine);
    var labelOrphans = data.labelOrphans;
    return {
        height: gfx.MARGIN // Top margin
            + (labelOrphans ? (gfx.em(1) + gfx.MARGIN) : 0) // Optional title
            + (lines * (gfx.NODE_HEIGHT + gfx.MARGIN)), // Issues, including last bottom margin.
        widthPx: gfx.NODE_WIDTH + ((maxColumns - 1) * (gfx.NODE_WIDTH + gfx.MARGIN)),
        draw_fn: VT.Graph.drawOrphans,
        layout_data: {
            issueKeys: _.sortBy(orphans, function (e) {
                return VT.Fn.counterFromIssueKey(e);
            }),
            issuesPerLine: issuesPerLine
        }
    };
};

VT.Graph.layoutSubgraph = function(subgraph, gfx, data) {
    // Describe graph nodes and edges
    var graph = new dagre.graphlib.Graph();
    graph.setGraph({
        nodesep: gfx.MARGIN,
        rankdir: VT.Graph.dagreRankDirForGraphDirection(data.graphDirection),
        ranksep: VT.Graph.isGraphDirectionHorizontal(data) ?
            gfx.H_MARGIN :
            gfx.V_MARGIN
    });
    graph.setDefaultEdgeLabel(function() { return {}; });
    _.each(subgraph.issues, function(issue) {
        graph.setNode(
            issue,
            {
                width: gfx.NODE_WIDTH,
                height: gfx.NODE_HEIGHT
            }
        );
    });

    subgraph.collapsedEdges = VT.Graph.calculateCollapsedEdges(subgraph.relations);
    _.each(subgraph.collapsedEdges, function(obj, v) {
        _.each(obj, function(collapsedEdge, w) {
            graph.setEdge(v, w, { collapsedEdge: collapsedEdge });
        });
    });

    // Layout the graph
    dagre.layout(graph);

    // Convert dagre (x, y) coordinates into SVG coordinates
    var HALF_NODE_WIDTH = Math.ceil(gfx.NODE_WIDTH / 2),
        HALF_NODE_HEIGHT = Math.ceil(gfx.NODE_HEIGHT / 2);
    function dagreXToDrawingX(key) {
        return Math.floor(gfx.MARGIN + (graph.node(key).x - HALF_NODE_WIDTH));
    }
    function dagreYToDrawingY(key) {
        return Math.floor(gfx.MARGIN + (graph.node(key).y - HALF_NODE_HEIGHT));
    }
    _.each(graph.nodes(), function(node) {
        var n = graph.node(node);
        n.x = dagreXToDrawingX(node);
        n.y = dagreYToDrawingY(node);
    });

    // Determine width and height
    var maxX = 0;
    var maxY = 0;
    _.each(graph.nodes(), function(node) {
        var n = graph.node(node);
        // Adjust dagre center points to node bounds
        maxX = Math.max(maxX, n.x + gfx.NODE_WIDTH);
        maxY = Math.max(maxY, n.y + gfx.NODE_HEIGHT);
    });

    // Determine sort key
    var sortkey = data.sortKeyForIssuesFn(subgraph.issues);

    return {
        widthPx: maxX,
        height: gfx.MARGIN + maxY,
        sortkey: sortkey,
        draw_fn: VT.Graph.drawSubgraph,
        layout_data: graph
    };
};

VT.Graph.orderLayers = function(gfx) {
    gfx.layers.backgroundLayer.back();
    gfx.layers.edgeLayer.front();
    gfx.layers.labelLayer.front();
    gfx.layers.issueLayer.front();
};

VT.Graph.resize = function(gfx) {
    if (gfx) {
        var viewportDims = gfx.viewportDimensionsHandler(gfx.displayListWidth, gfx.displayListHeight);

        var viewportMode = gfx.viewportModeFn();
        if (viewportMode == 'zoom') {
            var width  = viewportDims.viewportWidth;
            var height = viewportDims.viewportHeight;

            VT.Graph.SVG.size(gfx.paper, width, height);

            // Scale up the wallpaper to cover the viewport when zoom factor is at its minimum.
            var minZoomFactor = vivid.trace.jira.relation_graph.control_cluster_view.min_zoom_factor(
                gfx.displayListWidth, gfx.displayListHeight,
                viewportDims.viewportWidth, viewportDims.viewportHeight
            );
            VT.Graph.SVG.size(gfx.wallpaper, Math.ceil(width / minZoomFactor), Math.ceil(height / minZoomFactor));

            if (gfx.viewControlCluster) {
                gfx.viewControlCluster.resize(gfx.displayListWidth, gfx.displayListHeight, width, height);
            }
        } else {
            // Assuming :scroll
            var width = Math.floor(Math.max(
                viewportDims.viewportWidth, gfx.displayListWidth
            ));
            var height = Math.floor(Math.max(
                viewportDims.viewportHeight, gfx.displayListHeight
            ));

            VT.Graph.SVG.size(gfx.paper, width, height);
            VT.Graph.SVG.size(gfx.wallpaper, width, height);
        }
    }

    // Possibly standalone HTML (i.e. the Trace macro in Confluence)
    if (VT.Fn.isInFrame()) {
        var frameElement = window.frameElement;
        if (frameElement) {
            frameElement.style.height = window.self.document.body.scrollHeight + "px";
        }
    }
};

// Assumes viewport-mode is :scroll.
VT.Graph.scrollViewToIssue = function(issueKey, gfx) {
    if (issueKey) {
        var elementId = '#' + this.elementIdForNode(issueKey);
        var el = VT.Graph.SVG.getElementsByCSSSelector(gfx, elementId)[0];
        if (!_.isUndefined(el)) {
            var viewportDims = gfx.viewportDimensionsHandler(gfx.displayListWidth, gfx.displayListHeight);
            var bbox = VT.Graph.SVG.bbox(el);
            var scrollPosition = bbox.x + (bbox.width / 2) - (viewportDims.viewportWidth / 2);
            AJS.$('#vt-graph-container')[0].scrollLeft = scrollPosition;
        }
    }
};

VT.Graph.setHyperlinkToIssue = function(gfx, issueKey, data) {
    var issueUrl = VT.Fn.applicationBaseUrl() + '/browse/' + issueKey;
    var link = VT.Graph.SVG.link(data, gfx, issueUrl);
    if (_.isFunction(data.issueNavigationClickHandler)) {
        link.on('click', function (event) {
            event.preventDefault();

            data.issueNavigationClickHandler(issueKey, issueUrl);
        });
    }
    return link;
};

VT.Graph.tooltipTextForIssue = function(gfx, issue, el) {
    var tooltipText = issue[':jira.issue-field.system/key'] + ': ' + issue[':jira.issue-field.system/summary'];
    VT.Graph.SVG.tooltip(tooltipText, el);
    return tooltipText;
};

VT.Theme = {
    DefaultFieldWidth: 84,
    EdgeLabelOutlineColor: 'light-dark(white,#093560)',
    EdgeLabelOutlineRadius: 2.667,
    EdgeStrokeColor: 'light-dark(' + VT.ADG_Colors.MediumGray + ',#54bee3)',
    EdgeStrokeWidth: '1.7px',
    FontFamily: 'Arial, sans-serif',
    FontSizeInPx: 14,
    FontSizeWithUnits: '14px',
    ImageDimension: 16,
    InterFieldGap: 5,
    IssueBackgroundColor: 'light-dark(white,#052245)',
    // Desaturated variant indicates resolved issues
    IssueBackgroundResolvedColor: 'light-dark(#f0f0f0,#03162c)',
    IssueLozengeCornerRadiusInEms: 1,
    IssueStrokeColor: 'light-dark(#e0e0e0,#1b427a)',
    IssueStrokeWidth: '1.5px',
    LinkText: 'light-dark(#3b73af,rgb(87, 157, 255))',
    TextColor: 'light-dark(#333333,rgb(182, 194, 207))',
    UnspotlitCSSClassName: 'unspotlit',
    Wallpaper: 'light-dark(' + VT.ADG_Colors.LightGray + ',#011122)',
};
