/**
 * @license
 * Copyright 2022 Vivid Inc. and/or its affiliates.
 */

/*
 * Draws all elements of issue relation graphs using SVG.
 * Self-contained library then depends only on svg.js and the ``gfx'' data structure.
 *
 * Draw at integral coordinates so as not to degrade visual appearance.
 */

"use strict";

if (typeof VT === 'undefined') {
    var VT = {};
}
if (typeof VT.Graph === 'undefined') {
    VT.Graph = {};
}
if (typeof VT.Graph.SVG === 'undefined') {
    VT.Graph.SVG = {};
}

/**
 * SVG <foreignObject> to support multi-line text boxes of specific
 * width and height where overflowing lines are hidden.
 */
SVG.ForeignObject = SVG.invent({
   create   : 'foreignObject',
   inherit  : SVG.Shape,
   extend   : {
       appendChild: function (child, attrs) {
           var newChild = typeof (child) == 'string' ?
               document.createElement(child) : child
           if (typeof (attrs) == 'object') {
               for (const a in attrs) newChild[a] = attrs[a]
           }
           this.node.appendChild(newChild)
           return this
       },
       getChild   : function (index) {
           return this.node.childNodes[index]
       }
   },
   construct: {
       foreignObject: function (width, height) {
           return this.put(new SVG.ForeignObject).size(width, height);
       }
   }
});

/**
 * Add <title> elements, to provide supplementary information
 * via mouse-over tooltips.
 *
 * Referencing https://github.com/wout/svg.js/issues/147
 */
SVG.Title = SVG.invent({
    create: 'title',
    inherit: SVG.Element,
    extend: {
        addTo: function(parent) {
            parent.node.insertBefore(this.node, null);
            return this;
        },
        text: function(text) {
            while (this.node.firstChild) {
                this.node.removeChild(this.node.firstChild);
            }
            this.node.appendChild(document.createTextNode(text));
            return this;
        }
    },
    construct: {
        title: function(text) {
            return this.put(new SVG.Title).text(text);
        }
    }
});

/**
 * Constrain width of drawn elements to their stated widths.
 */
VT.Graph.SVG.abbreviateTextToFitWidthInPx = function(gfx, el, maxWidthPx) {
    if (!maxWidthPx) {
        // There is no specified width.
        return;
    }

    var actualWidthPx = el.node.getBBox().width;
    if (actualWidthPx <= maxWidthPx) {
        // The text already fits within the maximum width.
        return;
    }

    // The text exceeds the maximum width; shorten it.
    // Use bisection to approximate a width that fits the maximum width.
    var text = el.text();
    var a = 0;
    var b = text.length;
    // Limit iterations to prevent infinite loop
    var ITERATION_LIMIT = 20;
    var count = 0;
    while (count < ITERATION_LIMIT) {
        if (Math.abs(a - b) <= 1) {
            VT.Graph.SVG.tooltip(text, el);
            return;
        }
        var midpoint = Math.floor((b + a) / 2);
        var t = text.substring(0, midpoint);
        el.text(t + '...');
        var widthPx = el.node.getBBox().width;

        if (widthPx > maxWidthPx) {
            b = midpoint;
        } else {
            a = midpoint;
        }

        count++;
    }

    // If the foregoing was insufficient, use a clipping region instead.
    var bbox = el.node.getBBox();
    var rect = el.doc()
        .rect(maxWidthPx, bbox.height)
        .move(bbox.x, bbox.y);
    var clip = el.doc().clip().add(rect);
    el.clipWith(clip);
    VT.Graph.SVG.tooltip(text, el);
};

VT.Graph.SVG.addTo = function(coll) {
    function add(el) {
        if (el) {
            coll.add(el);
        }
    }
    for (var i = 1; i < arguments.length; i++) {
        var els = arguments[i];
        if (_.isArray(els)) {
            _.each(els, add);
        } else {
            add(els);
        }
    }
};

VT.Graph.SVG.alignText = function(el, alignment) {
    el.attr('text-anchor', alignment);
};

VT.Graph.SVG.assignElementId = function(el, elementId) {
    el.attr('id', elementId);
};

VT.Graph.SVG.bbox = function(el) {
    return el.node.getBBox();
};

VT.Graph.SVG.getElementsByCSSSelector = function(gfx, sel) {
    return gfx.paper.select(sel).members;
};

VT.Graph.SVG.group = function(gfx) {
    return gfx.paper.group();
};

/**
 * Supplied with an SVG document element, produces a self-contained HTML
 * variation suitable for export.
 */
VT.Graph.SVG.htmlOf = function(el, width, height) {
    var div = document.createElement('div');
    var replica = new SVG(div)
        .svg(el.instance.svg());

    // Resize the SVG to its exact dimensions
    replica.size(width, height);

    // Embed images directly into the document
    var imgs = replica.select('image').members;
    _.each(imgs, function(img) {
        var url = img.attr('xlink:href');
        var id = 'vt-cache-image-' + VT.Fn.hashString(url);
        var ref = document.getElementById(id);
        if (ref) {
            img.attr('xlink:href', ref.href.baseVal);
        }
    });

    // Produce an SVG based on valid XML with proper escaping of XML elements.
    var html = new XMLSerializer().serializeToString(div.firstChild);
    div.remove();
    return html;
};

VT.Graph.SVG.image = function(gfx, x, y, url, dimension, name) {
    // Create an SVG def for the image, for export
    var id = 'vt-cache-image-' + VT.Fn.hashString(url);
    if (!document.getElementById(id)) {
        var image = gfx.paper.defs().image();
        image.id(id);

        var xhr = new XMLHttpRequest();
        xhr.open('GET', url, true);
        xhr.responseType = 'blob';
        xhr.onload = function() {
            var blob = this.response;
            var reader = new window.FileReader();
            reader.readAsDataURL(blob);
            reader.onloadend = function() {
                var base64 = reader.result;
                image.load(base64).size(dimension, dimension);
            };
        };
        xhr.send();
    }

    var i = gfx.paper.image(url)
        .size(dimension, dimension)
        .move(x, y);
    VT.Graph.SVG.tooltip(name, i);

    return i;
};

VT.Graph.SVG.initializeDefs = function(gfx) {
    _.extend(gfx,
        VT.Graph.SVG.initializeMarkerDefs(gfx),
        VT.Graph.SVG.initializeFilterDefs(gfx)
    );
};

VT.Graph.SVG.initializeMarkerDefs = function(gfx) {
    var edgeEndMarker = gfx.paper.defs()
        .marker(4, 4).ref(3, 2);
    edgeEndMarker
        .path('M0,0L4,2L0,4z')
        .attr({
            fill: gfx.theme.EdgeStrokeColor
        });

    var edgeStartMarker = gfx.paper.defs()
        .marker(4,4).ref(1, 2);
    edgeStartMarker
        .path('M4,0L0,2L4,4z')
        .attr({
            fill: gfx.theme.EdgeStrokeColor
        });

    return { markers: {
        edgeEndMarker: edgeEndMarker,
        edgeStartMarker: edgeStartMarker
    }};
};

VT.Graph.SVG.initializeFilterDefs = function(gfx) {
    var edgeLabelOutlineFilter = gfx.paper.defs()
        .filter(function(add) {
            var f = add.flood(gfx.theme.EdgeLabelOutlineColor, 1);
            var cm = add
                .morphology('dilate', gfx.theme.EdgeLabelOutlineRadius);
            var c = add.composite(f, cm, 'in');
            add.blend(add.source, c, 'normal');
        });
    return { filters: {
        edgeLabelOutlineFilter: edgeLabelOutlineFilter
    }};
};

VT.Graph.SVG.issueCursor = function(gfx, issueEl) {
    var bbox = issueEl.node.getBBox();
    var HM = gfx.MARGIN * 0.888;

    var cursor = gfx.paper.rect()
        .move(Math.floor(bbox.x - HM), Math.floor(bbox.y - HM))
        .size(
            Math.ceil(bbox.width + HM * 2),
            Math.ceil(bbox.height + HM * 2))
        .radius(gfx.em(1));
    cursor.attr({
        // TODO Merge these into VT.Theme in relation-graph.js
        'fill': 'light-dark(#ddd,#0c345d)',
        'stroke': 'light-dark(#bbb,#0f4478)',
        'opacity': '0.333'
    });

    return cursor;
};

VT.Graph.SVG.issueLozenge = function(gfx, x, y, issue) {
    var rect = gfx.paper
        .rect(gfx.NODE_WIDTH, gfx.NODE_HEIGHT)
        .move(Math.floor(x), Math.floor(y))
        .rx(gfx.em(gfx.theme.IssueLozengeCornerRadiusInEms)).ry(gfx.em(gfx.theme.IssueLozengeCornerRadiusInEms));
    var isResolved = 'resolution' in issue;
    rect.attr({
        'fill': isResolved ?
            gfx.theme.IssueBackgroundResolvedColor :
            gfx.theme.IssueBackgroundColor,
        'stroke': gfx.theme.IssueStrokeColor,
        'stroke-width': gfx.theme.IssueStrokeWidth
    });
    return rect;
};

VT.Graph.SVG.link = function(data, gfx, url) {
    var l = gfx.paper.link(url);
    if (_.isString(data.issueLinkTargetOptions)) {
        l.target(data.issueLinkTargetOptions);
    }
    return l;
};

/**
 * SVG path drawing command that draws a line from the current
 * position to the specified coordinates.
 */
VT.Graph.SVG.p_line = function(x, y) {
    return 'L' + x + ',' + y;
};

/**
 * SVG path drawing command that moves the current pen position
 * without drawing.
 */
VT.Graph.SVG.p_move = function(x, y) {
    return 'M' + x + ',' + y;
};

VT.Graph.SVG.paper = function(node) {
    return SVG(node).size(0, 0).addClass('vt');
};

VT.Graph.SVG.relationEdge = function(gfx, layout_data, edge, y_offset) {
    var cmd = this.p_move(edge.coords.a.x, edge.coords.a.y + y_offset) +
        this.p_line(edge.coords.b.x, edge.coords.b.y + y_offset);
    var el = gfx.paper.path(cmd);

    el.attr({
        'stroke': gfx.theme.EdgeStrokeColor,
        'stroke-width': gfx.theme.EdgeStrokeWidth
    });

    var info = layout_data.edge(edge.v, edge.w);
    if (info.collapsedEdge.arrowheads[edge.v]) {
        el.attr('marker-start', gfx.markers.edgeStartMarker);
    }
    if (info.collapsedEdge.arrowheads[edge.w]) {
        el.attr('marker-end', gfx.markers.edgeEndMarker);
    }

    return el;
};

VT.Graph.SVG.relationLabel = function(gfx, text, tooltip, x, y) {
    var label = gfx.paper
        .text(text)
        .move(Math.floor(x), Math.floor(y))
        .font({anchor: 'middle'})
        .attr({
            'fill': gfx.theme.TextColor
        });
        VT.Graph.SVG.tooltip(tooltip, label);

    label.attr('filter', gfx.filters.edgeLabelOutlineFilter);
    return label;
};

VT.Graph.SVG.set = function(gfx) {
    return gfx.paper.set();
};

VT.Graph.SVG.size = function(el, width, height) {
    el.size(width, height);
};

VT.Graph.SVG.stylizeHyperlink = function(gfx, el) {
    el.fill(gfx.theme.LinkText);
};

VT.Graph.SVG.text = function(gfx, x, y, text, width, height) {
    if (height === undefined) {
        // No height was specified:
        // - Assume a single line of text.
        // - Strip all newlines from the text.
        // - Overflowing text is abbreviated and visually indicated with an ellipsis.
        var t  = text.replace(/[\r\n]/g, ' ');
        var el = gfx.paper
                    .text(t)
                    .attr({
                              'fill'       : gfx.theme.TextColor,
                              'font-family': gfx.theme.FontFamily,
                              'font-size'  : gfx.theme.FontSizeWithUnits,
                              'text-anchor': 'start'
                          })
                    .move(x, y);
        this.abbreviateTextToFitWidthInPx(gfx, el, width);
        VT.Graph.SVG.tooltip(text, el);
        return el;
    }
    else {
        var height = height + gfx.em(0.5); // Account for extenders in typeface.
        var el = gfx.paper
                    .foreignObject(width, height)
                    .move(x, y);
        el
            .appendChild("div",
                         {
                             innerText: text,
                             style    :
                                 'color: ' + gfx.theme.TextColor + ';' +
                                 'font-family: ' + gfx.theme.FontFamily + ';' +
                                 'font-size: ' + gfx.theme.FontSizeWithUnits + ';' +
                                 'height: ' + height + 'px;' +
                                 'line-height: 1.5em;' +
                                 'overflow: hidden;'
                         });
        VT.Graph.SVG.tooltip(text, el);
        return el;
    }
};

VT.Graph.SVG.textNull = function(gfx) {
    return gfx.paper
        .text('')
        .move(0, 0)
        .opacity(0);
};

/**
 * Browsers supported by Vivid Trace don't support 'text-decoration'.
 */
VT.Graph.SVG.textStrikethrough = function(gfx, el, color) {
    var bbox = el.node.getBBox();
    var y = Math.ceil(bbox.y + (bbox.height / 2));
    return gfx.paper
        .line(bbox.x, y, bbox.x + bbox.width, y)
        .stroke({
            color: color,
            width: gfx.theme.FontSizeInPx / 10
        });
};

VT.Graph.SVG.tooltip = function(text, parent) {
    var title = parent.doc().title(text);
    title.addTo(parent);
    return title;
};

VT.Graph.SVG.wallpaper = function(gfx) {
    var wallpaper = gfx.paper.rect();
    wallpaper.addClass("vt-relation-graph-wallpaper");
    wallpaper.fill({color: gfx.theme.Wallpaper});
    return wallpaper;
};
