/**
 * @license
 * Copyright 2025 Vivid Inc. and/or its affiliates.
 */

"use strict";

// TODO VT 2025.2: For artifact-types and issue-link-types, make very clear the distinction between enabling all types (updating dynamically whenever the available set changes, due to additions/changes/deletions of types) (a value of 'undefined') and intentionally setting none (the value of the empty set).
// TODO Mark Trace Studio dirty whenever there are unsaved changes. Helps prevent unintended data loss.
// TODO Upon receiving a trace configuration, ignore everything unknown. Upon saving a TC, work only with known values and merge/ignore with everything else unknown.

if (typeof VT == 'undefined') {
    var VT = {};
}
if (typeof VT.Trace == 'undefined') {
    VT.Trace = {};
}

/*
 * Implementation note: As a result of Backbone's design and perhaps my own lack of knowledge, when a user performs
 * an operation on a Backbone model/collection, the associated information is updated in Backbone's own data structures
 * prior to Backbone making the related REST API calls. When the REST API call reports an error, potentially
 * indicating that the change was not effected in Jira, from the user's perspective the UI's expression of application
 * state is no longer in sync with Jira.
 */

/**
 * Signals when the favorite status of a trace configuration has changed.
 */
VT.Trace.FAVORITE_EVENT = 'VividTraceFavorite';

/**
 * Occurs when the software is instructed to load a trace configuration.
 * The optional argument specifies the trace configuration ID, the absence of which indicates that a new trace is to be loaded (reset the UI to defaults).
 */
VT.Trace.LOAD_TRACE_EVENT = 'VividTraceLoadTrace';

/**
 * Signals when a trace configuration has been renamed.
 */
VT.Trace.RENAME_EVENT = 'VividTraceRename';

/**
 * Signals the need to resize the UI widgets as the viewport size may have changed.
 */
VT.Trace.RESIZE_EVENT = 'VividTraceResize';

/**
 * CLJS integration point.
 */
VT.Trace.SET_ARTIFACTTYPES_EVENT = 'VividTraceSetArtifactTypes';

/**
 * CLJS integration point.
 */
VT.Trace.SET_ISSUELINKTYPES_EVENT = 'VividTraceSetIssueLinkTypes';

/**
 * CLJS integration point.
 */
VT.Trace.SET_ITEMCARDLAYOUT_EVENT = 'VividTraceSetItemCardLayout';

/**
 * CLJS integration point.
 */
VT.Trace.SET_VIEWPORTMODE_EVENT = 'VividTraceSetViewportMode';

/**
 * CLJS integration point.
 */
VT.Trace.TRACE_CONFIGURATION_SAVED_EVENT = 'VividTraceTraceConfigurationSaved';

VT.Trace.exportData = function(issueRelationGraphView) {
    var el = AJS.$('#vt-graph-container').find('svg')[0];
    var dimensions = {
        width: issueRelationGraphView.gfx.displayListWidth,
        height: issueRelationGraphView.gfx.displayListHeight
    };
    var html = VT.Graph.SVG.htmlOf(el, dimensions.width, dimensions.height);
    var blob = new Blob([html], {type: 'image/svg+xml;charset=utf-8'});
    return _.extend({
            blob: blob
        },
        dimensions);
};

VT.Trace.exportFilename = function(extension) {
    var now = new Date;
    var datestamp = now.toJSON().replace(/T.*/, '');
    return 'VividTrace-' + datestamp + '.' + extension;
};

/**
 * Requirements:
 * Exported PNG is identical in appearance to graph shown on the display device.
 */
VT.Trace.exportPortableNetworkGraphic = function(issueRelationGraphView) {
    VT.Fn.trigger(VT.Graph.REMOVE_SPOTLIGHT_EVENT);

    VT.Fn.polyfill_HTMLCanvasElement_toBlob();

    var data = VT.Trace.exportData(issueRelationGraphView);

    var image = new Image;
    var objectUrl = URL.createObjectURL(data.blob);
    image.src = objectUrl;
    AJS.$(image).on('load', function() {
        var canvas = document.createElement('canvas');
        canvas.width = data.width;
        canvas.height = data.height;

        var context = canvas.getContext('2d');
        context.drawImage(image, 0, 0);
        URL.revokeObjectURL(objectUrl);

        canvas.toBlob(function(blob) {
            var filename = VT.Trace.exportFilename('png');
            vtSaveAs(blob, filename);
            canvas.remove();
        });
    });
};

/**
 * Requirements:
 * Exported SVG is identical in appearance to graph shown on the display device.
 * Issues and issue keys are hyperlinked with the correct fully-qualified URL.
 */
VT.Trace.exportScalableVectorGraphic = function(issueRelationGraphView) {
    VT.Fn.trigger(VT.Graph.REMOVE_SPOTLIGHT_EVENT);

    var data = VT.Trace.exportData(issueRelationGraphView);
    var filename = VT.Trace.exportFilename('svg');
    vtSaveAs(data.blob, filename);
};

VT.Trace.greedyDimensionsOf = function(sel) {
    var els = AJS.$(sel);
    if (_.size(els) >= 1) {
        var el = els[0];
        var bcr = el.getBoundingClientRect();
        return {
            el: el,
            greedyWidth: window.innerWidth - bcr.left,
            greedyHeight: window.innerHeight - bcr.top
        };
    }
};

VT.Trace.greedyDimensionsOfIssueRelationGraph = function() {
    var PADDING = 15;
    var dim = VT.Trace.greedyDimensionsOf('#vt-graph-container');
    if (dim) {
        return {
            greedyWidth: dim.greedyWidth - PADDING,
            greedyHeight: dim.greedyHeight - PADDING
        };
    }
};

VT.Trace.initialize = function(data) {
    // CLJS integration point.
    vivid.trace.jira.trace_studio.core.initialize__js(
        JSON.parse(data['artifactTypeMap'] || '{}'),
        JSON.parse(data['issueLinkTypeMap'] || '{}')
    );

    AJS.$('.vt-trace-studio').html(VT.Trace.Templates.traceStudio());

    if (_.isString(data.blankTraceConfiguration)) {
        data.blankTraceConfiguration = JSON.parse(data.blankTraceConfiguration);
    }
    if (_.isString(data.defaultTraceConfiguration)) {
        data.defaultTraceConfiguration = JSON.parse(data.defaultTraceConfiguration);
    }
    if (_.isString(data.initialTraceConfiguration)) {
        data.initialTraceConfiguration = JSON.parse(data.initialTraceConfiguration);
        if (_.isString(VT.Fn.get(data, 'initialTraceConfiguration.configuration'))) {
            data.initialTraceConfiguration.configuration = JSON.parse(data.initialTraceConfiguration.configuration);
        }
    }

    var traceConfiguration = VT.Trace.traceConfigurationFromUrlParams(data.blankTraceConfiguration);
    if (_.isEmpty(traceConfiguration)) {
        traceConfiguration = data.initialTraceConfiguration || data.defaultTraceConfiguration;
        // Layer the default configuration underneath the initial configuration,
        // effectively filling in missing values in the trace configuration.
        traceConfiguration.configuration = _.extend(
            data.defaultTraceConfiguration.configuration,
            traceConfiguration.configuration,
        );
    }

    var simplifiedRelationsSeedIssuesArg = VT.Trace.simplifyRelationsSeedIssuesArg(traceConfiguration.configuration.relationsSeedIssuesArg);

    var issueRelationGraphModel = new VT.Components.IssueRelationGraphModel(
        _.extend(
            data,
            {
                activeTraceConfigurationId: VT.Fn.get(data, 'initialTraceConfiguration.id'),
                issueLinkTargetOptions: '_blank',
                issues: data.issues || {},
                relations: data.relations || [],
                sortKeyForIssuesFn: VT.Trace.sortKeyForIssues
            },
            traceConfiguration.configuration,
            {
                // Interpret relationsSeedIssuesArg for direct use in the seed issues field.
                relationsSeedIssuesArg: simplifiedRelationsSeedIssuesArg
            }
        )
    );

    var issueRelationGraphView = new VT.Components.IssueRelationGraphView({
        el: '.vt-relation-graph',
        model: issueRelationGraphModel,
        fetchGraphDataDelegate: function(requestContext) {
            var artifactTypes = this.model.get('artifactTypes');
            var directions = this.model.get('directions');
            var distance = parseInt(this.model.get('distance'));
            var includeSeedIssues = this.model.get('includeSeedIssues');
            var issueLinkTypes = this.model.get('issueLinkTypes');
            var itemCardLayout = this.model.get('itemCardLayout');
            var relationsSeedIssuesArg = this.model.getRelationsSeedIssuesArg();

            // Requests are generated by user-driven events in time sequence, so this new request context
            // represents the most recent request; any older request context can be safely clobbered.
            this.model.set('requestContext', requestContext);

            var _this = this;
            var requestData =
                    _.extend({
                                 directions            : directions,
                                 distance              : !isNaN(distance) ? distance : undefined,
                                 includeSeedIssues     : includeSeedIssues,
                                 itemCardLayout        : itemCardLayout,
                                 relationsSeedIssuesArg: relationsSeedIssuesArg
                             });
            if (!_.isEmpty(artifactTypes)) {
                requestData['artifactTypes'] = artifactTypes;
            }
            if (!_.isEmpty(issueLinkTypes)) {
                requestData['issueLinkTypes'] = issueLinkTypes;
            }
            AJS.$.ajax({
                url: VT.Fn.applicationBaseUrl() + '/rest/vivid-trace/1/graph/relations',
                data: requestData,
                success: function(data) {
                    if (VT.Components.isCurrentRequestContext(_this.model, requestContext)) {
                        _this.model.unset('requestContext');

                        _this.model.set(data);
                        VT.Fn.trigger(VT.Components.REDRAW_IRG_EVENT);
                        VT.Trace.resizeIssueRelationGraph(issueRelationGraphView);
                    }
                },
                error: function(xhr) {
                    VT.Fn.showRestErrorMessage(
                        xhr
                    );

                    // Restore the IRG as it was to remove the appearance of a request being in-flight.
                    VT.Fn.trigger(VT.Components.REDRAW_IRG_EVENT);
                }
            });
        },
        viewportDimensions: VT.Trace.traceStudioViewportDimensions
    });

    var mountTraceStudioFn =
        _.partial(function(issueRelationGraphModel, event, traceConfigurationModel) {
            var acl = (! _.isUndefined(traceConfigurationModel) && traceConfigurationModel.get('acl')) ||
                (issueRelationGraphModel.get('initialTraceConfiguration') || {} )['acl'] ||
                [];
            vivid.trace.jira.trace_studio.core.mount({
                acl: acl,
                creation: (! _.isUndefined(traceConfigurationModel) && traceConfigurationModel.get('creation')),
                effectiveAccess: (! _.isUndefined(traceConfigurationModel) && traceConfigurationModel.get('effective-access')),
                favorite: (! _.isUndefined(traceConfigurationModel) && traceConfigurationModel.get('favorite').toString()),
                favoriteCount: (! _.isUndefined(traceConfigurationModel) && traceConfigurationModel.get('favorite-count').toString()),
                mountPointElementId: 'vt-trace-configuration-details-shell',
                name: (! _.isUndefined(traceConfigurationModel) && traceConfigurationModel.get('name')),
                traceConfigurationId: issueRelationGraphModel.get('activeTraceConfigurationId')
            });
        }, issueRelationGraphModel);

    VT.Fn.bind(VT.Trace.LOAD_TRACE_EVENT,
        _.partial(function(issueRelationGraphModel, event, traceConfigurationModel) {
                issueRelationGraphModel.unset('artifactTypes');
                issueRelationGraphModel.unset('issueLinkTypes');
                if (_.isUndefined(traceConfigurationModel)) {
                    // Feature: Creating a new trace reloads the model with defaults, inducing
                    // a redraw to a blank "No issues" state. Deselect the active saved
                    // trace, as necessary.
                    issueRelationGraphModel.set(
                        _.extend(
                            {
                                activeTraceConfigurationId: undefined,
                                issues: {},
                                relations: [],
                                messages: {}
                            },
                            issueRelationGraphModel.get('blankTraceConfiguration').configuration
                        )
                    );
                    VT.Fn.trigger(VT.Components.FETCH_IRG_EVENT);
                } else {
                    // Feature: Load a trace configuration.
                    // Attempt to update the trace configuration by fetching it from the server.
                    issueRelationGraphModel.set(
                        _.extend(
                            {
                                activeTraceConfigurationId: traceConfigurationModel.id,
                                issues: {},
                                relations: [],
                                messages: {}
                            },
                            // Layer the default configuration underneath the initial configuration,
                            // effectively filling in missing values in the trace configuration.
                            issueRelationGraphModel.get('blankTraceConfiguration').configuration,
                            JSON.parse(traceConfigurationModel.get('configuration'))
                        )
                    );
                    VT.Fn.trigger(VT.Components.FETCH_IRG_EVENT);
                }
            },
            issueRelationGraphModel));

    VT.Trace.renderTraceConfigurationShell(
        issueRelationGraphModel,
        function() {
            return VT.Trace.serializeTraceConfiguration(issueRelationGraphModel);
        },
        VT.Trace.serializeTraceConfigurationKeys,
        data.initialTraceConfiguration
    );

    var resizeHandler = VT.Trace.resizeWireHandler(issueRelationGraphView);
    VT.Fn.bind(VT.Trace.RESIZE_EVENT, resizeHandler);
    // Resize once now
    resizeHandler();

    VT.Trace.wireTraceOperations(
        '.vt-trace-operations',
        issueRelationGraphModel,
        issueRelationGraphView
    );

    function addTraceModule(elementIdStem, title) {
        AJS.$('.vt-trace-modules').append(
            VT.Trace.Templates.traceModule({
                elementIdStem: elementIdStem,
                title: title,
            })
        );
    }

    // Relations parameters
    addTraceModule(
        'vt-relations-parameters',
        AJS.I18n.getText('vivid.trace.relations-parameters.label')
    );
    var relationsParametersView = new VT.Trace.RelationsParametersView({
        el: '#vt-relations-parameters-module > .mod-content',
        model: issueRelationGraphModel
    });
    relationsParametersView.render();

    // Display Options
    addTraceModule(
        'vt-display-options',
        AJS.I18n.getText('vivid.trace.issue-relation-graph.display-options-group')
    );
    var displayOptionsView = new VT.Trace.DisplayOptionsView({
        el: '#vt-display-options-module > .mod-content',
        model: issueRelationGraphModel
    });
    displayOptionsView.render();

    // Issue link types
    addTraceModule(
        'vt-issue-link-types',
        AJS.I18n.getText('vivid.trace.configure-project.issue-link-types-tab'),
    );
    // CLJS integration point.
    vivid.trace.jira.trace_studio.core.add_module_title_bar__js(
        '#vt-issue-link-types-module .title-bar', // sel
        'vivid.trace.configure-project.issue-link-types-tab', // module-i18n-key
        'vivid.trace.configure-project.manage-issue-link-types', // admin-i18n-key
        data.isUserJiraAdmin ? VT.Fn.applicationBaseUrl() + '/secure/admin/ViewLinkTypes!default.jspa' : undefined,
        'vivid.trace.configure-project.issue-link-types.description.html', // help-i18n-key
        AJS.I18n.getText('vivid.trace.documentation.url') + '/trace-studio.html#issue-link-types',
    );
    VT.Components.renderIssueLinkTypeEditor("#vt-issue-link-types-module > .mod-content");
    function copyIssueLinkTypesToBackbone(event, issueLinkTypesJsonString) {
        issueRelationGraphModel.set({issueLinkTypes: issueLinkTypesJsonString});
        VT.Fn.trigger(VT.Components.FETCH_IRG_EVENT);
    }
    VT.Fn.bind(VT.Trace.SET_ISSUELINKTYPES_EVENT, copyIssueLinkTypesToBackbone);
    function loadIssueLinkTypesFromBackbone(event) {
        vivid.trace.jira.trace_studio.core.load_issue_link_types__js(
            JSON.parse(issueRelationGraphModel.get('issueLinkTypes') || '[]'),
            JSON.parse(issueRelationGraphModel.get('issueLinkTypeMap') || '{}'),
        );
    }
    VT.Fn.bind(VT.Trace.LOAD_TRACE_EVENT, loadIssueLinkTypesFromBackbone);
    loadIssueLinkTypesFromBackbone(null);

    // Artifact types
    addTraceModule(
        'vt-artifact-types',
        AJS.I18n.getText('vivid.trace.configure-project.artifact-types-tab'),
    );
    // CLJS integration point.
    vivid.trace.jira.trace_studio.core.add_module_title_bar__js(
        '#vt-artifact-types-module .title-bar',
        'vivid.trace.configure-project.artifact-types-tab',
        'vivid.trace.configure-project.manage-issue-types',
        data.isUserJiraAdmin ? VT.Fn.applicationBaseUrl() + '/secure/admin/ViewIssueTypes.jspa' : undefined,
        'vivid.trace.configure-project.artifact-types.description.html',
        AJS.I18n.getText('vivid.trace.documentation.url') + '/trace-studio.html#artifact-types',
    );
    // CLJS integration point.
    VT.Components.renderArtifactTypeEditor("#vt-artifact-types-module > .mod-content");
    function copyArtifactTypesToBackbone(event, artifactTypesJsonString) {
        issueRelationGraphModel.set({artifactTypes: artifactTypesJsonString});
        VT.Fn.trigger(VT.Components.FETCH_IRG_EVENT);
    }
    VT.Fn.bind(VT.Trace.SET_ARTIFACTTYPES_EVENT, copyArtifactTypesToBackbone);
    function loadArtifactTypesFromBackbone(event) {
        vivid.trace.jira.trace_studio.core.load_artifact_types__js(
            JSON.parse(issueRelationGraphModel.get('artifactTypes') || '[]'),
            JSON.parse(issueRelationGraphModel.get('artifactTypeMap') || '{}'),
        );
    }
    VT.Fn.bind(VT.Trace.LOAD_TRACE_EVENT, loadArtifactTypesFromBackbone);
    loadArtifactTypesFromBackbone(null);

    // Item card layout
    addTraceModule(
        'vt-item-card-layout',
        AJS.I18n.getText('vivid.trace.item-card-layout.label')
    );
    // CLJS integration point.
    vivid.trace.jira.trace_studio.core.add_module_title_bar__js(
        "#vt-item-card-layout-module .title-bar",
        'vivid.trace.item-card-layout.label',
        'vivid.trace.configure-project.manage-custom-fields',
        data.isUserJiraAdmin ? VT.Fn.applicationBaseUrl() + '/secure/admin/ViewCustomFields.jspa' : undefined,
    );
    VT.Components.renderItemCardLayoutEditor("#vt-item-card-layout-module > .mod-content");
    function copyItemCardLayoutToBackbone(event, itemCardLayoutEdnString) {
        issueRelationGraphModel.set({itemCardLayout: itemCardLayoutEdnString});
        VT.Fn.trigger(VT.Components.FETCH_IRG_EVENT);
    }
    VT.Fn.bind(VT.Trace.SET_ITEMCARDLAYOUT_EVENT, copyItemCardLayoutToBackbone);
    function loadItemCardLayoutFromBackbone(event) {
        vivid.trace.jira.trace_studio.core.load_item_card_layout__js(
            issueRelationGraphModel.get('itemCardLayout')
        );
    }
    VT.Fn.bind(VT.Trace.LOAD_TRACE_EVENT, loadItemCardLayoutFromBackbone)
    loadItemCardLayoutFromBackbone(null);

    // Feature: Set initial focus on the seed issues input field.
    AJS.$('#vt-relations-seed-issues-arg').focus();

    if (!VT.Fn.hasErrorMessages(data.messages)) {
        issueRelationGraphView.fetchGraphData();
    }

    // Start with these tools buttons disabled. Both rely on the generated JQL, which in
    // the case of an initialized trace UI is the JQL "issue in relations()" which is
    // different than the initialized Trace UI state that a user actually sees, which is
    // no JQL and no issues.
    var hasRelationsSeedIssuesArg = function(issueRelationGraphModel) {
        var seedIssues = issueRelationGraphModel.getRelationsSeedIssuesArg();
        return !_.isEmpty(seedIssues);
    };
    var AFFECTED_TOOL_BUTTON_SELECTORS = '#vt-bulk-change-button, #vt-list-issues-button';
    VT.Components.auiDisableButton(AFFECTED_TOOL_BUTTON_SELECTORS);
    // As soon as a seed issues query comes into being, then enable the buttons.
    VT.Fn.bind(
        VT.Components.REDRAW_IRG_EVENT,
        _.partial(function(issueRelationGraphModel) {
            if (hasRelationsSeedIssuesArg(issueRelationGraphModel)) {
                VT.Components.auiEnableButton(AFFECTED_TOOL_BUTTON_SELECTORS);
            } else {
                VT.Components.auiDisableButton(AFFECTED_TOOL_BUTTON_SELECTORS);
            }
        }, issueRelationGraphModel)
    );
    // Reset to disabled when a new trace is started.
    VT.Fn.bind(
        VT.Trace.LOAD_TRACE_EVENT,
        _.partial(function(issueRelationGraphModel, event, traceConfigurationModel) {
            if (_.isUndefined(traceConfigurationModel)) {
                VT.Components.auiDisableButton(AFFECTED_TOOL_BUTTON_SELECTORS);
            }
        }, issueRelationGraphModel)
    );

    // CLJS integration point.
    // Activate the CLJS-based parts now.
    mountTraceStudioFn();
    // Refresh the CLJS-based parts in response to the following events:
    VT.Fn.bind(
        VT.Components.FETCH_IRG_EVENT,
        mountTraceStudioFn
    );
    VT.Fn.bind(
        VT.Trace.LOAD_TRACE_EVENT,
        mountTraceStudioFn
    );
    VT.Fn.bind(
        VT.Trace.TRACE_CONFIGURATION_SAVED_EVENT,
        mountTraceStudioFn
    );

    // Activate view splitters
    VTSplit({
                columnGutters: [{
                    track  : 1,
                    element: document.querySelector('#vt-trace-configurations-split')
                }],
                minSize      : 14,
                onDrag       : function (_direction, _track, _gridTemplateStyle) {
                    VT.Fn.trigger(VT.Trace.RESIZE_EVENT);
                }
            });
    VTSplit({
                columnGutters: [{
                    track  : 1,
                    element: document.querySelector('#vt-trace-modules-split')
                }],
                minSize      : 14,
                onDrag       : function (_direction, _track, _gridTemplateStyle) {
                    VT.Fn.trigger(VT.Trace.RESIZE_EVENT);
                }
            });
};

VT.Trace.DirectionsView = VT.Backbone.View.extend({
    click: function(event) {
        var direction = event.target.value;
        var enabled = event.target.checked;
        var curState = JSON.parse(
            this.model.get('directions')
        );
        var newState = enabled ?
            _.union(curState, [direction]) :
            _.without(curState, direction);

        // Note: I would rather see this view re-render()ed after the model is updated so that the view reflects the
        // true value of the state. This code merely hopes that the view is more or less synchronized with model state.

        if (_.size(newState) <= 0) {
            // This is a safety measure as the UI is not intended to allow directions to be empty.
            // Enable all directions: in UI, in newState.
            var els = this.$el.find('input.vt-directions');
            els
                .removeAttr('disabled')
                .attr('checked', 'checked');
            newState = _.map(els, function(el) {
                return el.value;
            });
        } else {
            // Resume free selection.
            this.$el.find('input.vt-directions').removeAttr('disabled');
        }

        this.model.set({directions: JSON.stringify(newState)});
        VT.Fn.trigger(VT.Components.FETCH_IRG_EVENT);
    },
    initialize: function() {
        this.model.on('change:directions', _.bind(this.render, this));
    },
    render: function() {
        var directions = JSON.parse(this.model.get('directions'));
        var soleEnabledDirection = _.size(directions) === 1 && directions[0];
        this.$el.html(this.template({
            directions: this.toTemplateParams(),
            // Feature: When only one direction is enabled, disable it's UI control to prevent
            // the effective directions from becoming empty.
            soleEnabledDirection: soleEnabledDirection
        }));
        this.$el.find('input.vt-directions')
            .off('click.vt')
            .on('click.vt', _.bind(this.click, this));
    },
    template: VT.Trace.Templates.directionsInputGroup,
    toTemplateParams: function() {
        return _.reduce(
            JSON.parse(
                this.model.get('directions')
            ),
            function (memo, v) {
                memo[v] = true;
                return memo;
            },
            {}
        );
    }
});

VT.Trace.jqlQueryFromModel = function(model) {
    function relationsArg_seedIssues(model) {
        return model.getRelationsSeedIssuesArg();
    }
    function relationsArg_artifactType(model) {
        var map = _.keys(JSON.parse(model.get('artifactTypeMap')));
        if (! _.isEmpty(model.get('artifactTypes'))) {
            var enabled = JSON.parse(model.get('artifactTypes'));
            return !_.isEmpty(_.difference(map, enabled)) ?
                '"artifactType IN (' + enabled.join(', ') + ')"' :
                // Default value of the artifactType parameter for the relations() suite of JQL functions
                undefined;
        }
    }
    function relationsArg_direction(model) {
        var v = JSON.parse(model.get('directions'));
        return _.size(v) >= 4 ?
            // Default value of the direction parameter for the relations() suite of JQL functions
            undefined :
        '"direction in (' + v.join(', ') + ')"';
    }
    function relationsArg_distance(model) {
        var v = model.get('distance');
        return _.size(v) >= 1 ?
        '"distance = ' + v + '"' :
            // Default value of the distance parameter for the relations() suite of JQL functions
            undefined;
    }
    function relationsArg_inclusive(model) {
        var v = model.get('includeSeedIssues');
        return String(v).toLowerCase() === 'true' ?
            // Default value of the inclusive parameter for the relations() suite of JQL functions
            undefined :
        '"inclusive = false"';
    }
    function relationsArg_issueLinkType(model) {
        var map = _.keys(JSON.parse(model.get('issueLinkTypeMap')));
        if (! _.isEmpty(model.get('issueLinkTypes'))) {
            var enabled = JSON.parse(model.get('issueLinkTypes'));
            return !_.isEmpty(_.difference(map, enabled)) ?
                '"issueLinkType IN (' + enabled.join(', ') + ')"' :
                // Default value of the issueLinkType parameter for the relations() suite of JQL functions
                undefined;
        }
    }

    return 'issue IN relations(' +
        _.filter(
            [
                relationsArg_artifactType(model),
                relationsArg_seedIssues(model),
                relationsArg_direction(model),
                relationsArg_distance(model),
                relationsArg_inclusive(model),
                relationsArg_issueLinkType(model)
            ],
            function(el) { return _.size(el) >= 1; }
        )
            .join(', ')
        + ')';
};

VT.Trace.PrimitiveCheckboxView = VT.Backbone.View.extend({
    click: function() {
        var newState = this.model.get(this.options.modelKey) ? false : true; // Invert

        // Update the model.
        this.model.set(this.options.modelKey, newState);
        JIRA.trigger(this.options.modelSetEventKey);
    },
    initialize: function() {
        this.model.on('change:' + this.options.modelKey, _.bind(this.render, this));
    },
    render: function() {
        this.$el.html(this.options.template(this.model.toJSON()));
        this.$el.find(this.options.checkboxEl)
            .off('click.vt')
            .on('click.vt', _.bind(this.click, this));
    }
});

VT.Trace.RelationsSeedIssueArgSettingView = VT.Backbone.View.extend({
    getArg: function() {
        return VT.Trace.simplifyRelationsSeedIssuesArg(
            this.$el.find('#vt-relations-seed-issues-arg')[0].value.trim()
        );
    },
    initialize: function() {
        this.model.on('change:relationsSeedIssuesArg', _.bind(this.render, this));
        _.bindAll(this, 'getArg', 'keypress', 'keyup', 'renderRelationsSeedIssuesArgIcon');
        this.render();
    },
    keypress: function(event) {
        if (event.keyCode === 13) {
            return this.setRelationsSeedIssueArg(event);
        }
    },
    keyup: function() {
        this.renderRelationsSeedIssuesArgIcon();
    },
    render: function() {
        this.$el.html(this.template(this.model.toJSON()));
        this.$el.find('#vt-relations-seed-issues-arg').on('keypress', this.keypress);
        this.$el.find('#vt-relations-seed-issues-arg').on('keyup', this.keyup);
        this.$el.find('#vt-search-button').on('click', this.setRelationsSeedIssueArg);
        // Feature: Resize the seed issues input field to fit all of its content.
        AJS.$('#vt-relations-seed-issues-arg')[0].style.height = AJS.$('#vt-relations-seed-issues-arg').prop('scrollHeight') + 'px';
        this.renderRelationsSeedIssuesArgIcon();
    },
    renderRelationsSeedIssuesArgIcon: function() {
        var relationsSeedIssuesArg = this.getArg();
        var _this = this;

        function setArgIcon(statusClass) {
            var iconEl = _this.$el.find('.vt-relations-seed-issues-arg-icon');
            iconEl.removeClass('jqlgood jqlerror');
            if (!_.isUndefined(statusClass)) {
                iconEl.addClass(statusClass ? 'jqlgood' : 'jqlerror');
            }
        }

        if (_.size(relationsSeedIssuesArg) <= 0) {
            setArgIcon();
        } else if (_.size(VT.Components.asIssueKeys(relationsSeedIssuesArg)) >= 1) {
            setArgIcon(true);
        } else {
            AJS.$.ajax({
                url: VT.Fn.applicationBaseUrl() + '/rest/vivid-trace/1/system/jql',
                data: _.extend({
                    jql: relationsSeedIssuesArg
                }),
                success: function(data) {
                    setArgIcon(data.valid);
                },
                error: function(xhr) {
                    VT.Fn.showRestErrorMessage(
                        xhr
                    );
                }
            });
        }
    },
    setRelationsSeedIssueArg: function(event) {
        event.preventDefault();
        var relationsSeedIssuesArg = this.getArg();
        this.model.set({relationsSeedIssuesArg: relationsSeedIssuesArg});
        VT.Fn.trigger(VT.Components.FETCH_IRG_EVENT);
        return false;
    },
    template: VT.Trace.Templates.relationsSeedIssueArgSetting
});

VT.Trace.DisplayOptionsView = VT.Backbone.View.extend({
    render: function() {
        this.$el.html(this.template());

        var graphDirectionView = new VT.Components.GraphDirectionView({
            el: this.$el.find('.vt-graph-direction-container'),
            model: this.model
        });
        graphDirectionView.render();

        var labelOrphansSettingView = new VT.Trace.PrimitiveCheckboxView({
            checkboxEl: '.vt-label-orphans',
            el: this.$el.find('.vt-label-orphans-container'),
            model: this.model,
            modelKey: 'labelOrphans',
            modelSetEventKey: VT.Components.REDRAW_IRG_EVENT,
            template: VT.Trace.Templates.labelOrphansSetting
        });
        labelOrphansSettingView.render();

        var showRelationshipLabelsView = new VT.Components.ShowRelationshipLabelsSettingView({
            el: this.$el.find('.vt-show-relationship-labels-container'),
            model: this.model,
            template: VT.Trace.Templates.showRelationshipLabels
        });
        showRelationshipLabelsView.render();
    },
    template: VT.Trace.Templates.displayOptions
});

VT.Trace.RelationsParametersView = VT.Backbone.View.extend({
    render: function() {
        this.$el.html(this.template());

        var relationsSeedIssueArgSettingView = new VT.Trace.RelationsSeedIssueArgSettingView({
            el: this.$el.find('.vt-relations-seed-issues-arg-container'),
            model: this.model
        });
        relationsSeedIssueArgSettingView.render();

        var directionsView = new VT.Trace.DirectionsView({
            el: this.$el.find('.vt-directions-container'),
            model: this.model
        });
        directionsView.render();

        var distanceSettingView = new VT.Components.DistanceSettingView({
            el: this.$el.find('.vt-distance-setting-container'),
            model: this.model
        });
        distanceSettingView.render();

        var includeSeedIssuesSettingView = new VT.Trace.PrimitiveCheckboxView({
            checkboxEl: '.vt-include-seed-issues',
            el: this.$el.find('.vt-include-seed-issues-setting-container'),
            model: this.model,
            modelKey: 'includeSeedIssues',
            modelSetEventKey: VT.Components.FETCH_IRG_EVENT,
            template: VT.Trace.Templates.includeSeedIssuesSetting
        });
        includeSeedIssuesSettingView.render();
    },
    template: VT.Trace.Templates.relationsOptions
});

VT.Trace.renderTraceConfigurationShell = function(
    issueRelationGraphModel,
    serializeFn,
    serializeKeys,
    initialTraceConfiguration
) {
    var configurationsCollection = new VT.Trace.TraceConfigurations();

    if (!_.isUndefined(initialTraceConfiguration)) {
        var i = new VT.Trace.TraceConfiguration(initialTraceConfiguration);
        configurationsCollection.add(i);
    }

    var configurationsView = new VT.Trace.TraceConfigurationsView({
        collection: configurationsCollection,
        el: '.vt-trace-configurations',
        issueRelationGraphModel: issueRelationGraphModel
    });
    configurationsView.render();

    var headerView = new VT.Trace.TraceConfigurationHeaderView({
        el: '.vt-active-trace-session',
        configurationsCollection: configurationsCollection,
        model: issueRelationGraphModel,
        serializeFn: serializeFn,
        serializeKeys: serializeKeys
    });
    headerView.render();

    // Feature: When the active saved trace is deleted, re-initialize the trace UI.
    function loadNewTraceWhenActiveTraceIsDeleted(a) {
        var activeId = issueRelationGraphModel.get('activeTraceConfigurationId');
        if (!_.isUndefined(a.id) && (a.id === activeId)) {
            VT.Fn.trigger(VT.Trace.LOAD_TRACE_EVENT);
        }
    }
    configurationsCollection.on('remove', loadNewTraceWhenActiveTraceIsDeleted);
    // CLJS integration point.
    var renameModelFn = function(event, traceConfigurationId,  name) {
        var model = traceConfigurationId && configurationsCollection.get(traceConfigurationId);
        model.set({
            name: name
        });
        VT.Fn.trigger(VT.Trace.TRACE_CONFIGURATION_SAVED_EVENT, model);
        configurationsView.render();
    };
    // Trace configuration renaming event
    VT.Fn.bind(
        VT.Trace.RENAME_EVENT,
        _.bind(renameModelFn, this)
    );
};

VT.Trace.resizeIssueRelationGraph = function(issueRelationGraphView) {
    var dim = VT.Trace.greedyDimensionsOfIssueRelationGraph();
    if (dim) {
        var opts = {
            viewportWidth: dim.greedyWidth,
            viewportHeight: dim.greedyHeight,
            sizeCSS: true // TODO
        };
        issueRelationGraphView.resizeViewport(opts);
    }
};

VT.Trace.resizeTraceConfigurations = function() {
    var PADDING = 15;
    var dim = VT.Trace.greedyDimensionsOf('.vt-trace-configurations .vt-saved-traces');
    if (dim) {
        AJS.$(dim.el).css('height', dim.greedyHeight - PADDING);
    }
};

VT.Trace.resizeTraceModules = function() {
    var PADDING = 15;
    var dim = VT.Trace.greedyDimensionsOf('.vt-trace-modules');
    if (dim) {
        AJS.$(dim.el).css('height', dim.greedyHeight - PADDING);
    }
};

VT.Trace.resizeWireHandler = function(issueRelationGraphView) {
    function h(issueRelationGraphView) {
        VT.Trace.resizeTraceConfigurations();
        VT.Trace.resizeTraceModules();
        VT.Trace.resizeIssueRelationGraph(issueRelationGraphView);
    }
    var handler = _.partial(h, issueRelationGraphView);
    AJS.$(window).resize(handler);

    return handler;
};

VT.Trace.simplifyRelationsSeedIssuesArg = function(arg) {
    if (_.size(arg) <= 0) {
        return arg;
    }
    var unquoted = VT.Fn.unquote(arg).trim();
    var matches = unquoted.match(/^\s*jql\s*=(.*)/i);
    if (_.size(matches) >= 2) {
        return matches[1].trim();
    } else {
        return arg;
    }
};

VT.Trace.TraceConfigurationHeaderView = VT.Backbone.View.extend({
    discardChanges: function() {
        var savedModel = this.getTraceConfigurationModel();
        if (!savedModel) {
            return;
        }
        savedModel.fetch()
            .success(function() {
                VT.Fn.trigger(VT.Trace.LOAD_TRACE_EVENT, savedModel);
            })
            .error(function(xhr) {
                VT.Fn.showRestErrorMessage(xhr);
            });
    },
    getTraceConfigurationModel: function() {
        var id = this.model.get('activeTraceConfigurationId');
        return id && this.options.configurationsCollection.get(id);
    },
    initialize: function() {
        var bound_render = _.bind(this.render, this);

        VT.Fn.bind(VT.Trace.TRACE_CONFIGURATION_SAVED_EVENT, bound_render);
        this.model.on('change:activeTraceConfigurationId', bound_render);
        this.options.configurationsCollection.on('change', bound_render);

        var _m = this.model;
        _.each(this.options.serializeKeys, function(key) {
            _m.on('change:' + key, bound_render);
        });
    },
    isDirty: function() {
        var savedModel = this.getTraceConfigurationModel();
        if (!savedModel) {
            return;
        }
        var savedConfiguration = savedModel.get('configuration');
        if (_.isString(savedConfiguration)) {
            savedConfiguration = JSON.parse(savedConfiguration);
        }
        var activeConfiguration = this.options.serializeFn();
        return !_.isEqual(
            savedConfiguration || {},
            JSON.parse(activeConfiguration || '{}')
        );
    },
    render: function() {
        var savedModel = this.getTraceConfigurationModel();
        var isSaveEnabled = savedModel && _.contains(['editor', 'owner'], savedModel.get("effective-access"));

        this.$el.html(this.template({
            // Feature: Only authenticated users can save traces.
            isUserAuthenticated: this.model.get('isUserAuthenticated'),
            name: (savedModel && savedModel.get('name')) || AJS.I18n.getText('vivid.trace.phrase.trace')
        }));

        this.$el.find('.vt-saved-trace-operations').html(this.isDirty() ?
                VT.Trace.Templates.savedTraceSaveAsButton({
                    saveDisabled: !isSaveEnabled
                }) :
                VT.Trace.Templates.savedTraceSaveButton()
        );
        if (isSaveEnabled) {
            this.$el.find('.vt-trace-configuration-save')
                .off('click.vt')
                .on('click.vt', _.bind(this.save, this));
        }
        this.$el.find('.vt-trace-configuration-save-as')
            .off('click.vt')
            .on('click.vt', _.bind(this.saveAs, this));
        this.$el.find('.vt-trace-configuration-discard-changes')
            .off('click.vt')
            .on('click.vt', _.bind(this.discardChanges, this));
    },
    save: function() {
        VT.Fn.uiElementActuated(event);
        var savedModel = this.getTraceConfigurationModel();
        var serializeFn = this.options.serializeFn;
        if (!savedModel) {
            return;
        }
        // Update just configuration as the current user might not have
        // permission to change other aspects such as name.
        savedModel.save({}, {
            contentType: 'application/json',
            data: JSON.stringify({configuration: this.options.serializeFn()})
        })
            .success(function() {
                savedModel.set({
                    configuration: serializeFn()
                });
                JIRA.Messages.showSuccessMsg(
                    AJS.I18n.getText(
                        'vivid.trace.trace-configuration.save-success',
                        '<b>' + savedModel.get('name') + '</b>'
                    )
                );
                VT.Fn.trigger(VT.Trace.TRACE_CONFIGURATION_SAVED_EVENT, savedModel);
            })
            .error(function(xhr) {
                VT.Fn.showRestErrorMessage(xhr);
            });
        return false;
    },
    saveAs: function(event) {
        VT.Fn.uiElementActuated(event);
        VT.Trace.traceConfigurationSaveAsDialog(this.options.serializeFn);
        return false;
    },
    template: VT.Trace.Templates.savedTrace
});

VT.Trace.serializeTraceConfiguration = function(model) {
    return JSON.stringify(model.pick(VT.Trace.serializeTraceConfigurationKeys));
};

VT.Trace.serializeTraceConfigurationKeys = [
    'artifactTypes',
    'directions',
    'distance',
    'includeSeedIssues',
    'graphDirection',
    'itemCardLayout',
    'issueLinkTypes',
    'labelOrphans',
    'relationsSeedIssuesArg',
    'showRelationshipLabels'
];

VT.Trace.sortKeyForIssues = function(issues) {
    return issues.sort()[0];
};

VT.Trace.TraceConfiguration = VT.Backbone.Model.extend({
    urlRoot: VT.Fn.applicationBaseUrl() + '/rest/vivid-trace/1/trace'
});

VT.Trace.TraceConfigurationView = VT.Backbone.View.extend({
    render: function() {
        var model = this.model;
        var isDeleteEnabled = model && _.contains(['owner'], model.get('effective-access'));
        var isRenameEnabled = model && _.contains(['owner'], model.get('effective-access'));
        var isUnfavoriteEnabled = model && model.get('favorite');

        var html = this.template(
            _.extend({
                    deleteDisabled: !isDeleteEnabled,
                    renameDisabled: !isRenameEnabled,
                    unfavoriteDisabled: !isUnfavoriteEnabled,
                    url: VT.Fn.applicationBaseUrl() + '/secure/trace/?trace=' + model.get('id')
                },
                model.toJSON()
            ));
        this.$el.html(html).attr('data-id', this.model.id);

        var isActive = model.id === this.options.activeTraceConfigurationId;
        if (isActive) {
            this.$el.addClass('vt-active-trace-configuration');
        }

        // Feature: Recall a trace configuration by clicking on it, or can be opened
        // in new user agent tabs and windows.
        this.$el.find('.vt-trace-configuration-name')
            .off('click.vt')
            .on('click.vt', function(event) {
                VT.Fn.uiElementActuated(event);
                model.fetch()
                    .success(function() {
                        VT.Fn.trigger(VT.Trace.LOAD_TRACE_EVENT, model);
                    })
                    .error(function(xhr) {
                        VT.Fn.showRestErrorMessage(xhr);
                    });
                return false;
            });

        // Feature: When the mouse moves away from the saved trace and into its drop-down menu,
        // the saved trace entry stays highlighted.
        function tag(model) {
            return AJS.$('.vt-saved-traces li[data-id="' + model.id + '"]');
        }
        this.$el.find('.aui-dropdown2').on({
            'aui-dropdown2-show': _.bind(function() {
                tag(this.model).addClass('vt-active');
            }, this),
            'aui-dropdown2-hide': _.bind(function() {
                tag(this.model).removeClass('vt-active');
            }, this)
        });

        this.$el.find('.vt-trace-configuration-copy')
            .off('click.vt')
            .on('click.vt', this.model, VT.Trace.traceConfigurationDuplicateDialog);
        if (isDeleteEnabled) {
            this.$el.find('.vt-trace-configuration-delete')
                .off('click.vt')
                .on('click.vt', this.model, VT.Trace.traceConfigurationDeleteDialog);
        }
        if (isUnfavoriteEnabled) {
            this.$el.find('.vt-trace-configuration-unfavorite')
                .off('click.vt')
                .on('click.vt', this.model, _.partial(vivid.trace.jira.trace_studio.core.unfavorite_action__js, model.id));
        }
        if (isRenameEnabled) {
            this.$el.find('.vt-trace-configuration-rename')
                .off('click.vt')
                .on('click.vt', this.model, VT.Trace.traceConfigurationRenameDialog);
        }

        return this;
    },
    className: 'vt-saved-trace',
    tagName: 'li',
    template: VT.Trace.Templates.traceConfiguration
});

VT.Trace.TraceConfigurations = VT.Backbone.Collection.extend({
    addModel: function(event, model) {
        this.add(model);
    },
    comparator: function(a, b) {
        return VT.Fn.caseInsensitiveStringComparator(
            a.get('name'),
            b.get('name')
        );
    },
    initialize: function() {
        VT.Fn.bind(VT.Trace.TRACE_CONFIGURATION_SAVED_EVENT, _.bind(this.addModel, this)); // TODO
    },
    model: VT.Trace.TraceConfiguration,
    url: VT.Fn.applicationBaseUrl() + '/rest/vivid-trace/1/trace?favorite=true',
    parse: function(data) {
        return _.values(data["trace-configurations"]);
    }
});

VT.Trace.TraceConfigurationsView = VT.Backbone.View.extend({
    initialize: function() {
        this.collection.on('sync remove', _.bind(this.renderSavedTraces, this));
        this.options.issueRelationGraphModel.on('change:activeTraceConfigurationId', _.bind(this.renderSavedTraces, this));
        this.collection.fetch()
            .error(function(xhr) {
                VT.Fn.showRestErrorMessage(xhr);
            });


        VT.Fn.bind(VT.Trace.FAVORITE_EVENT, _.bind(this.refetchData, this));
    },
    refetchData: function(event) {
        this.collection.fetch()
            .error(function(xhr) {
                VT.Fn.showRestErrorMessage(xhr);
            });
    },
    newTrace: function(event) {
        VT.Fn.uiElementActuated(event);
        VT.Fn.trigger(VT.Trace.LOAD_TRACE_EVENT);
        return false;
    },
    render: function() {
        var html = this.template();
        this.$el.html(html);

        this.$el.find('.vt-new-trace-button')
            .off('click.vt')
            .on('click.vt', _.bind(this.newTrace, this));
        this.$el.find('.vt-find-traces-action')
            .off('click.vt')
            .on('click.vt', vivid.trace.jira.trace_studio.core.find_traces_action__js);

        // Feature: List of trace configurations
        this.renderSavedTraces();

        return this;
    },
    renderSavedTraces: function() {
        var el = this.$el.find('.vt-saved-traces');
        el.empty();
        var models = this.collection.sort().models;
        if (_.size(models) <= 0) {
            el.html(VT.Trace.Templates.traceConfigurationsNoSavedTraces());
        } else {
            var activeTraceConfigurationId = this.options.issueRelationGraphModel.get('activeTraceConfigurationId');
            _.each(models, function (model) {
                var view = new VT.Trace.TraceConfigurationView({
                    activeTraceConfigurationId: activeTraceConfigurationId,
                    model: model
                });
                var html = view.render().el;
                el.append(html);
            });
        }
    },
    template: VT.Trace.Templates.traceConfigurations
});

VT.Trace.traceConfigurationDeleteDialog = function(event) {
    var model = event.data;

    function delete_action_fn() {
        model.destroy({
            error: function(model, xhr) {
                VT.Fn.showRestErrorMessage(xhr);
            }
        });
    }

    var trace_configuration = {
        creation: model.get('creation'),
        "favorite-count": model.get('favorite-count'),
        name: model.get('name'),
    };
    vivid.trace.jira.add_on_trace_configuration_management.view.delete_confirmation_dialog_js(
        trace_configuration,
        delete_action_fn
    );
};

VT.Trace.traceConfigurationDialogGetNameInputValue = function(dialog) {
    var name = AJS.$('#' + dialog.id + ' input[id="name"]').val();
    return name.trim();
};

VT.Trace.traceConfigurationDialogSelectNameField = function(dialog) {
    // Feature: Select the text to speed editing.
    dialog.popup.element.find('#name').select();
};

VT.Trace.traceConfigurationDuplicateDialog = function(event) {
    var model = event.data;

    function submitDialog(dialog) {
        var name = VT.Trace.traceConfigurationDialogGetNameInputValue(dialog);
        // Fetch latest configuration from server.
        model.fetch()
            .success(function() {
                var duplicate = new VT.Trace.TraceConfiguration({
                    configuration: model.get('configuration'),
                    name: name
                });
                duplicate.save()
                    .error(function(xhr) {
                        VT.Fn.showRestErrorMessage(xhr);
                    });
                dialog.remove();
                VT.Fn.trigger(VT.Trace.TRACE_CONFIGURATION_SAVED_EVENT, duplicate);
            })
            .error(function(xhr) {
                VT.Fn.showRestErrorMessage(xhr);
            });
    }

    var name = model.get('name');
    var dialog = VT.Trace.traceConfigurationPrimitiveActionDialog({
        idFragment: 'copy',
        header: AJS.I18n.getText('vivid.trace.trace-configuration.duplicate-trace-header', name),
        panel: VT.Trace.Templates.traceConfigurationName({
            name: AJS.I18n.getText('vivid.phrase.copy-of', name)
        }),
        submitLabel: AJS.I18n.getText('vivid.phrase.duplicate'),
        submitFn: submitDialog
    });

    VT.Trace.traceConfigurationDialogSelectNameField(dialog);
};

VT.Trace.traceConfigurationFromUrlParams = function (blankTraceConfiguration) {
    const params           = new URLSearchParams(document.location.search);
    const rawConfiguration = {
        artifactTypes         : params.get('artifact-types'),
        directions            : params.get('directions'),
        distance              : params.get('distance'),
        graphDirection        : params.get('graph-direction'),
        includeSeedIssues     : params.get('include-seed-issues'),
        issueLinkTypes        : params.get('issue-link-types'),
        itemCardLayout        : params.get('item-card-layout'),
        // TODO As with label-orphans and others, the user needs a mechanism to specify an exact value (like bool false or true) that expressly overrides the defaults.
        // VT 2025.1 This field is not carried across from contextual traces: labelOrphans : params.get('label-orphans'),
        relationsSeedIssuesArg: params.get('seed-issues-jql-query'),
        showRelationshipLabels: params.get('show-relationship-labels'),
    }

    const conf    = Object.fromEntries(Object.entries(rawConfiguration).filter(([_, v]) => v));
    const hasData = Object.keys(conf).length >= 1;
    if (hasData) {
        const blank = blankTraceConfiguration?.configuration;
        return {configuration: {...blank, ...conf}};
    }
}

VT.Trace.traceConfigurationPrimitiveActionDialog = function(options) {
    var dialog = new AJS.Dialog({
        closeOnOutsideClick: true,
        id: 'vt-trace-configuration-' + options.idFragment + '-dialog',
        width: 540
    });
    function removeDialog() {
        dialog.remove();
    }
    dialog.addHeader(options.header);
    dialog.addPanel('Panel', options.panel, 'vt-message');
    dialog.addButton(options.submitLabel, options.submitFn, 'vt-' + options.idFragment + '-submit-button');
    dialog.addCancel(AJS.I18n.getText('vivid.phrase.cancel'), removeDialog, 'vt-cancel-button');

    dialog.show();
    dialog.updateHeight();

    var inputField = dialog.popup.element.find('#name');
    var submitButton = dialog.popup.element.find('.vt-' + options.idFragment + '-submit-button');

    // Name field validation
    function keyPress(event) {
        var key = event.keyCode || event.which;
        if (key === 13) {
            event.preventDefault();
            submitButton.trigger('click');
            return false;
        }
        return true;
    }

    function validateInput() {
        if (_.size(inputField) <= 0) {
            return;
        }
        var rawValue = inputField[0].value;
        var value = rawValue.trim();
        var valid = value !== undefined && _.size(value) >= 1;
        if (valid) {
            inputField.removeClass('vt-validation-error');
            submitButton.removeAttr('disabled');
        } else {
            inputField.addClass('vt-validation-error');
            submitButton.attr('disabled', 'disabled');
        }
    }
    dialog.popup.element.find('#name')
        .on('keyup', function() {
            validateInput();
        })
        .on('keypress', function(event) {
            keyPress(event);
        });

    // Allow the submit button to visually appear disabled.
    submitButton
        .removeClass('button-panel-button')
        .addClass('aui-button')
        .addClass('vt-disableable-button-panel-button');

    // Validate input once now to set button state correctly on start.
    validateInput();

    return dialog;
};

VT.Trace.traceConfigurationRenameDialog = function(event) {
    var model = event.data;

    function submitDialog(dialog) {
        var name = VT.Trace.traceConfigurationDialogGetNameInputValue(dialog);
        model.set({name: name});
        model.save()
            .error(function(xhr) {
                VT.Fn.showRestErrorMessage(xhr);
            });

        dialog.remove();
        VT.Fn.trigger(VT.Trace.TRACE_CONFIGURATION_SAVED_EVENT, model);
    }

    var name = model.get('name');
    var dialog = VT.Trace.traceConfigurationPrimitiveActionDialog({
        idFragment: 'rename',
        header: AJS.I18n.getText('vivid.trace.trace-configuration.rename-trace-header', name),
        panel: VT.Trace.Templates.traceConfigurationName({
            name: name
        }),
        submitLabel: AJS.I18n.getText('vivid.phrase.rename'),
        submitFn: submitDialog
    });

    VT.Trace.traceConfigurationDialogSelectNameField(dialog);
};

VT.Trace.traceConfigurationSaveAsDialog = function(serializeFn) {
    function submitDialog(dialog) {
        var name = VT.Trace.traceConfigurationDialogGetNameInputValue(dialog);
        var model = new VT.Trace.TraceConfiguration({
            configuration: serializeFn(),
            name: name
        });
        model.save()
            .success(function() {
                dialog.remove();
                VT.Fn.trigger(VT.Trace.TRACE_CONFIGURATION_SAVED_EVENT, model);
                VT.Fn.trigger(VT.Trace.LOAD_TRACE_EVENT, model);
            })
            .error(function(xhr) {
                VT.Fn.showRestErrorMessage(xhr);
            });
    }
    var dialog = VT.Trace.traceConfigurationPrimitiveActionDialog({
        idFragment: 'save-as',
        header: AJS.I18n.getText('vivid.trace.trace-configuration.save-trace-header'),
        panel: VT.Trace.Templates.traceConfigurationName({
            name: AJS.$(".vt-saved-trace-name").text()
        }),
        submitLabel: AJS.I18n.getText('vivid.phrase.save'),
        submitFn: submitDialog
    });
    VT.Trace.traceConfigurationDialogSelectNameField(dialog);
};

VT.Trace.wireTraceOperations = function(sel, issueRelationGraphModel, issueRelationGraphView) {
    AJS.$(sel).html(VT.Trace.Templates.traceOperations());

    // CLJS integration point.
    vivid.trace.jira.trace_studio.core.viewport_port_view__js("#vt-trace-viewport-mode-button-shell");

    AJS.$('#vt-trace-refresh-workspace-button')
        .attr({
            title: AJS.I18n.getText('vivid.trace.trace.refresh-workspace.description')
        })
        .on('click', function() {
            issueRelationGraphView.fetchGraphData();
        });

    function capability(sel, fn) {
        AJS.$(sel).on('click', fn);
    }
    AJS.$('#vt-trace-export-button').attr({
        title: AJS.I18n.getText('vivid.trace.export.description')
    });
    capability(
        '#vt-trace-export-png-button',
        _.partial(VT.Trace.exportPortableNetworkGraphic, issueRelationGraphView)
    );
    capability(
        '#vt-trace-export-svg-button',
        _.partial(VT.Trace.exportScalableVectorGraphic, issueRelationGraphView)
    );

    // When these buttons are clicked, first calculate their URLs based on the issue relation graph model state.

    VT.Components.wireButtonClick('#vt-bulk-change-button', function() {
        VT.Components.wireBulkChangeButton(
            '#vt-bulk-change-button',
            VT.Trace.jqlQueryFromModel(issueRelationGraphModel)
        );
        return true;
    });

    VT.Components.wireButtonClick('#vt-list-issues-button', function() {
        VT.Components.wireListIssuesButton(
            '#vt-list-issues-button',
            VT.Trace.jqlQueryFromModel(issueRelationGraphModel)
        );
        return true;
    });

};

VT.Trace.traceStudioViewportDimensions = function() {
    var dim = VT.Trace.greedyDimensionsOfIssueRelationGraph();
    return {
        viewportWidth: Math.ceil(dim.greedyWidth),
        viewportHeight: Math.ceil(dim.greedyHeight)
    };
};
