/**
 * @license
 * Copyright 2022 Vivid Inc. and/or its affiliates.
 */

"use strict";

if (typeof VT == 'undefined') {
    var VT = {};
}
if (typeof VT.Fn == 'undefined') {
    VT.Fn = {};
}

VT.Fn.applicationBaseUrl = function() {
    // If the proxy provided the application's base URL, use that.
    var applicationBaseUrl = VT.Fn.get(VT, '__proxy.application_base_url');
    if (typeof applicationBaseUrl != 'undefined') {
        return applicationBaseUrl;
    }
    // Otherwise, use the URL provided on the page generated by the
    // host application.
    var q = document.querySelector("meta[name='ajs-context-path']");
    return _.isObject(q) ? q.getAttribute('content') : '';
};

VT.Fn.allErrorMessages = function(messages) {
    return _.reduce(messages, function(memo, value, key) {
        if (VT.Fn.isErrorMessageCodePredicate(key)) {
            memo[key] = value;
        }
        return memo;
    }, {});
};

VT.Fn.allWarningMessages = function(messages) {
    return _.reduce(messages, function(memo, value, key) {
        if (VT.Fn.isWarningMessageCodePredicate(key)) {
            memo[key] = value;
        }
        return memo;
    }, {});
};

VT.Fn.caseInsensitiveStringComparator = function(a, b) {
    var x = a.toLowerCase();
    var y = b.toLowerCase();
    if (x < y) {
        return -1;
    } else if (x === y) {
        return 0;
    }
    return 1;
};

VT.Fn.clearElement = function(sel) {
    var el = AJS.$(sel)[0];
    if (el) {
        while (el.firstChild) {
            el.removeChild(el.firstChild);
        }
    }
    return el;
};

VT.Fn.counterFromIssueKey = function(issueKey) {
    return parseInt(issueKey.split("-")[1]);
};

/*
 * Gets the property value at path in object.
 */
VT.Fn.get = function(object, path, defaultValue) {
    var result = object == null ? undefined : VT.Fn._get(object, path);
    return result === undefined ? defaultValue : result;
};

/*
 * Referencing: http://stackoverflow.com/questions/6491463/accessing-nested-javascript-objects-with-string-key
 */
VT.Fn._get = function(o, s) {
    s = s.replace(/\[(\w+)\]/g, '.$1'); // convert indexes to properties
    s = s.replace(/^\./, '');           // strip a leading dot
    var a = s.split('.');
    for (var i = 0, n = a.length; i < n; ++i) {
        var k = a[i];
        if (_.isObject(o) && k in o) {
            o = o[k];
        } else {
            return;
        }
    }
    return o;
};

VT.Fn.hasErrorMessages = function(messages) {
    return _.some(messages, function(value, key) {
        return VT.Fn.isErrorMessageCodePredicate(key);
    });
};

/*
 * Referencing http://stackoverflow.com/questions/7616461/generate-a-hash-from-string-in-javascript-jquery
 */
VT.Fn.hashString = function(str) {
    var hash = 0, i, chr, len;
    if (str.length === 0) return hash;
    for (i = 0, len = str.length; i < len; i++) {
        chr   = str.charCodeAt(i);
        hash  = ((hash << 5) - hash) + chr;
        hash |= 0; // Convert to 32bit integer
    }
    return hash;
};

VT.Fn.isErrorMessageCodePredicate = function(messageCode) {
    var code = /^VTE\-\d+/.exec(messageCode);
    return _.size(code) >= 1;
};

/**
 * The standalone HTML code uses this to resize itself once its
 * content has been rendered.
 * @returns {boolean} when the current frame is not the top-level frame
 */
VT.Fn.isInFrame = function() {
    try {
        return window.self !== window.top;
    } catch (e) {
        return true;
    }
}

VT.Fn.isQuoted = function(s) {
    if (_.size(s) < 2) {
        return false;
    }
    var len = s.length;
    return (s[0] === "'" && s[len - 1] === "'") || (s[0] === '"' && s[len - 1] === '"');
};

VT.Fn.issueKeyComparator = function(a, b) {
    var x = a.split('-');
    var y = b.split('-');
    // Compare first on the project key.
    if (x[0] < y[0]) {
        return -1;
    } else if (x[0] > y[0]) {
        return 1;
    } else {
        // And if the project keys are equal, then on the issue IDs.
        return parseInt(x[1]) - parseInt(y[1]);
    }
};

VT.Fn.isWarningMessageCodePredicate = function(messageCode) {
    var code = /^VTW\-\d+$/.exec(messageCode);
    return _.size(code) >= 1;
};

VT.Fn.lengthOfLine = function(a, b) {
    return Math.sqrt( Math.pow(a.x - b.x, 2) + Math.pow(a.y - b.y, 2) );
};

VT.Fn.polyfill_HTMLCanvasElement_toBlob = function() {
    // Referencing https://developer.mozilla.org/en-US/docs/Web/API/HTMLCanvasElement/toBlob
    if (!HTMLCanvasElement.prototype.toBlob) {
        Object.defineProperty(HTMLCanvasElement.prototype, 'toBlob', {
            value: function (callback, type, quality) {

                var binStr = atob( this.toDataURL(type, quality).split(',')[1] ),
                    len = binStr.length,
                    arr = new Uint8Array(len);

                for (var i=0; i<len; i++ ) {
                    arr[i] = binStr.charCodeAt(i);
                }

                callback( new Blob( [arr], {type: type || 'image/png'} ) );
            }
        });
    }
};

VT.Fn.projectKey = function(issueKey) {
    var idx = issueKey && issueKey.indexOf('-');
    if (idx >= 1) return issueKey.substr(0, idx);
};

VT.Fn.removeDisabledAttribute = function() {
    // Convert arguments to a plain array.
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Functions/arguments
    var args = Array.prototype.slice.call(arguments);
    _.each(args, function(el) {
        el.removeAttr('disabled');
    });
};

VT.Fn.renderMessage = function(code, v) {
    if (_.isString(v)) {
        // Treat the value as the message string
        return v;
    }
    if (_.isObject(v)) {
        return v.message;
    }
    // Last alternative
    return code;
};

VT.Fn.scale = function(vector, factor) {
    return { x: vector.x * factor, y: vector.y * factor };
};

VT.Fn.setDisabledAttribute = function() {
    // Convert arguments to a plain array.
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Functions/arguments
    var args = Array.prototype.slice.call(arguments);
    _.each(args, function(el) {
        el.attr('disabled', 'disabled');
    });
};

VT.Fn.showMessage = function(messageObj) {
    var message = "<p><b>" + messageObj.message + "</b></p>";

    if (messageObj.supplementaryMessage) {
        message += "<p>" + messageObj.supplementaryMessage + "</p>";
    }

    var fn = messageObj.type == 'warning' ?
        JIRA.Messages.showWarningMsg :
        JIRA.Messages.showErrorMsg;
    fn(
        message,
        {
            timeout: -1
        }
    ).css({
        position: 'fixed',
        left: '50%',
        marginLeft: '-10%',
        top: '20px'
    });
};

VT.Fn.showRestErrorMessage = function(xhr, messageOptions) {
    // REST request timed out
    if (xhr.status == 0 && xhr.statusText && xhr.statusText.toLowerCase() == 'timeout') {
        VT.Fn.showVTE24Message(
            AJS.I18n.getText(
                'vivid.phrase.technical-note',
                AJS.I18n.getText('vivid.phrase.request-timed-out')
            ),
            messageOptions
        );
        return;
    }

    // Message(s) returned from server
    if (xhr.responseText) {
        var json = JSON.parse(xhr.responseText);
        if (_.isObject(json.messages)) {
            _.each(_.values(json.messages), function(message) {
                VT.Fn.showMessage(message);
            });
            if (_.size(json.messages) >= 1) {
                return;
            }
        }
    }

    // Treat all other errors as internal errors
    VT.Fn.showMessage({
        code: 'VTE-19',
        message: AJS.I18n.getText(
            'vivid.trace.error.vte-19-internal-error',
            '' // Blank the '{0}' placeholder in the I18n resource string
        ),
        supplementaryMessage: '(See the JavaScript console in your web browser\'s development tools for the XML HTTP Response object "xhr".)',
        type: 'error'
    });
    console.error('vivid.trace VT.Fn.showRestErrorMessage() xhr', xhr);
};

VT.Fn.showVTE24Message = function(detail, opts) {
    opts = opts || {};

    var message = "<p><b>" + AJS.I18n.getText('vivid.trace.error.vte-24-jira-communication-error') + "</b></p><p>";
    if (opts.changesNotSaved) {
        message += AJS.I18n.getText('vivid.phrase.changes-not-being-saved') + ' ';
    }
    message += AJS.I18n.getText('vivid.trace.phrase.contact-your-jira-administrators-if-problem-persists') + "</p>";

    if (detail) {
        message += "<p>" + detail + "</p>";
    }

    JIRA.Messages.showErrorMsg(
        message,
        {
            timeout: -1
        }
    ).css({
        position: 'fixed',
        left: '50%',
        marginLeft: '-10%',
        top: '20px'
    });
};

VT.Fn.translate = function(point, vector) {
    return { x: point.x + vector.x, y: point.y + vector.y };
};

/**
 * When a UI element is actuated, call this to prevent UI elements from displaying stale activation and selection state.
 */
VT.Fn.uiElementActuated = function(event) {
    // JavaScript pattern
    if (event) {
        event.preventDefault();
    }
    // A visible AUI dropdown might not react to clicks on buttons outside of its scope; hide all dropdowns now.
    AJS.$('.aui-dropdown2').each(
        function(index, dropdown) {
            // Jira 6.3.13 doesn't seem to have these functions.
            if (_.isFunction(dropdown.isVisible) && _.isFunction(dropdown.hide)) {
                if (dropdown.isVisible()) {
                    dropdown.hide();
                }
            }
        }
    );
};

VT.Fn.unitVectorFor = function(a, b) {
    var length = VT.Fn.lengthOfLine(a, b);
    return { x: (b.x - a.x) / length, y: (b.y - a.y) / length };
};

VT.Fn.unquote = function(s) {
    var trimmed = s.trim();
    return VT.Fn.isQuoted(trimmed) ? trimmed.slice(1, -1) : s;
};

/*
 * Fast UUID generator, RFC4122 version 4 compliant.
 * Call uuid.generate() to generate new UUIDs.
 *
 * author Jeff Ward (jcward.com).
 * license MIT license
 * link http://stackoverflow.com/questions/105034/how-to-create-a-guid-uuid-in-javascript/21963136#21963136
 **/
VT.Fn.uuid = (function() {
    var self = {};
    var lut = []; for (var i=0; i<256; i++) { lut[i] = (i<16?'0':'')+(i).toString(16); }
    self.generate = function() {
        var d0 = Math.random()*0xffffffff|0;
        var d1 = Math.random()*0xffffffff|0;
        var d2 = Math.random()*0xffffffff|0;
        var d3 = Math.random()*0xffffffff|0;
        return lut[d0&0xff]+lut[d0>>8&0xff]+lut[d0>>16&0xff]+lut[d0>>24&0xff]+'-'+
            lut[d1&0xff]+lut[d1>>8&0xff]+'-'+lut[d1>>16&0x0f|0x40]+lut[d1>>24&0xff]+'-'+
            lut[d2&0x3f|0x80]+lut[d2>>8&0xff]+'-'+lut[d2>>16&0xff]+lut[d2>>24&0xff]+
            lut[d3&0xff]+lut[d3>>8&0xff]+lut[d3>>16&0xff]+lut[d3>>24&0xff];
    };
    return self;
})();


//
// Bind and trigger: Cross-product replacements for JIRA.bind() et. al.
//
// Referencing atlassian-jira-6.3-source/jira-project/jira-components/jira-webapp/src/main/webapp/static/util/events.js
// Jira 6.3.0's JIRA.bind() calls jQuery(document) whereas these functions use the AJS-wrapped variant of jQuery.
//

VT.Fn.bind = function(types, data, fn) {
    return AJS.$(document).bind(types, data, fn);
};

VT.Fn.one = function(evt, handler) {
    return AJS.$(document).one(evt, handler);
};

VT.Fn.trigger = function(evt, args) {
    return AJS.$(document).trigger(evt, args)
};

VT.Fn.unbind = function(evt, args) {
    return AJS.$(document).unbind(evt, args);
};
